<?php
/**
 * @package	HikaShop for Joomla!
 * @version	3.1.0
 * @author	hikashop.com
 * @copyright	(C) 2010-2017 HIKARI SOFTWARE. All rights reserved.
 * @license	GNU/GPLv3 http://www.gnu.org/licenses/gpl-3.0.html
 */

use Stripe\Exception\ApiErrorException;

defined('_JEXEC') or die('Restricted access');

require_once(__DIR__ . '/vendor/autoload.php');

class StripeOrder {
	/**
	 * @var StripeCart
	 */
	protected $cart;

	/**
	 * @var StripePaymentIntentData
	 */
	protected $paymentIntent;

	/**
	 * @var \Stripe\Checkout\Session
	 */
	protected $session;

	/**
	 * @var object
	 */
	protected $order;

	public $currency = 'eur';

	public function __construct($order)
	{
		$this->order = $order;
	}

	public static function create($order): self
	{
		return new self($order);
	}

	public function createCart(): self
	{
		$cart_items = array_map(function ($item) {
			return new StripeCartItem($item);
		}, $this->order->cart->products);

		$this->cart = new StripeCart(...$cart_items);

		return $this;
	}

	private function setStripeCustomer($customer, $isValid = true)
	{
		$params = [];
		if ($isValid && isset($customer->user_params->stripe_customer_id)) {
			// Use saved Stripe ID for checkout
			$params['customer'] = $customer->user_params->stripe_customer_id;
		} else if ($customer->user_email) {
			// Use HikaShop Mail preferred
			$params['customer_email'] = $customer->user_email;
		} else if ($customer->email) {
			// fallback to Joomla Mail
			$params['customer_email'] = $customer->email;
		}

		return $params;
	}

	public function addAdditionalItems(): self
	{
		if (!$this->order->cart->additional) return $this;

		$cart_items = array_map(function ($item) {
			$item->quantity = 1;

			return new StripeCartItem($item);
		}, $this->order->cart->additional);

		$this->cart->addItems(...$cart_items);

		return $this;
	}

	private function addAdditionalCosts(string $name, float $amount, float $tax): self
	{
		$item = new StripeCartItem((object) [
			'order_product_name' => $name,
			'order_product_price' => round($amount - $tax, 3),
			'order_product_tax' => $tax,
			'order_product_quantity' => 1,
		]);

		$this->cart->append($item);

		return $this;
	}

	public function addShippingCostConditional(): self
	{
		if ($this->cart === null) {
			throw new Exception('Please first create a StripeCart');
		}

		// skip if no shipping cost are set or 0
		if (empty($this->order->order_shipping_price) || !bccomp($this->order->order_shipping_price, 0, 5)) {
			return $this;
		}

		return $this->addAdditionalCosts(JText::_('HIKASHOP_SHIPPING'), $this->order->order_shipping_price, $this->order->order_shipping_tax);
	}

	public function addPaymentCostConditional(): self
	{
		if ($this->cart === null) {
			throw new Exception('Please first create a StripeCart');
		}

		// skip if no shipping cost are set or 0
		if (empty($this->order->order_payment_price) || !bccomp($this->order->order_payment_price, 0, 5)) {
			return $this;
		}

		return $this->addAdditionalCosts(JText::_('HIKASHOP_PAYMENT'), $this->order->order_payment_price, $this->order->order_payment_price);
	}

	public function createPaymentIntent(string $address_type): self
	{
		$this->paymentIntent = new StripePaymentIntentData($this->order->cart->$address_type, $this->order);

		return $this;
	}

	public function createStripeSession($itemId): self
	{
		['customer' => $customer, 'order_id' => $order_id] = get_object_vars($this->order);

		$params = [
			'payment_method_types' => ['card'],
			'line_items' => json_decode(json_encode($this->cart), true),
			'success_url' => HIKASHOP_LIVE.'index.php?option=com_hikashop&ctrl=checkout&task=after_end&order_id='. $order_id . $itemId,
			'cancel_url' => HIKASHOP_LIVE.'index.php?option=com_hikashop&ctrl=order&task=cancel_order&order_id='. $order_id . $itemId,
			'client_reference_id' => $order_id,
			'payment_intent_data' => json_decode(json_encode($this->paymentIntent),true),
		];

		$params = array_merge($params, $this->setStripeCustomer($customer));

		$this->session = \Stripe\Checkout\Session::create($params);

		return $this;
	}

	public function getCart(): StripeCart
	{
		return $this->cart;
	}

	public function getPaymentIntent(): StripePaymentIntentData
	{
		return $this->paymentIntent;
	}

	public function getSession(): \Stripe\Checkout\Session
	{
		return $this->session;
	}

	public function getHikaOrder()
	{
		return $this->order;
	}
}

class StripeCartItem {
    public $amount;
    public $name;
    public $description;
    public $currency = 'eur';
    public $quantity = 1;
    public $images;

    public function __construct(object $item)
    {
        $this->name = substr(strip_tags($item->order_product_name), 0, 127);
        $this->amount = intval(round(($item->order_product_price + $item->order_product_tax) * 100));
        $this->quantity = intval($item->order_product_quantity);

        $this->images = array_map(function($img) {
            return HIKASHOP_LIVE.'images/com_hikashop/upload/'.$img->file_path;
        }, $item->images);
    }
}

class StripeCart extends ArrayIterator {
    public function __construct(StripeCartItem ...$items)
    {
        // filter free products because stripe doesn't accept them.
        $items = array_values(array_filter($items, function($item) {
            return $item->amount > 1; // must be at least one cent.
        }));

        parent::__construct($items);
    }
    public function current() : StripeCartItem
    {
        return parent::current();
    }
    public function offsetGet($offset) : StripeCartItem
    {
        return parent::offsetGet($offset);
    }

    public function getTotal()
    {
        $amounts = array_map(function(StripeCartItem $item) {
            return $item->quantity * $item->amount;
        }, iterator_to_array($this));
        return array_sum($amounts);
    }

    public function addItems(StripeCartItem ...$items)
    {
	    foreach ($items as $item) {
		    parent::append($item);
    	}
    }
}

class StripeAddress {
    public $line1;
    public $line2;
    public $postal_code;
    public $state;
    public $country;

    public function __construct(object $address)
    {
        $this->line1 = $address->address_street;
        $this->line2 = $address->address_street2;
        $this->postal_code = $address->address_post_code;

        $this->state = @$address->address_state->zone_code_3;
        $this->country = @$address->address_country->zone_code_2;
    }
}

class StripeShippingIntent {
    public $name;
    public $address;

    public function __construct(object $address)
    {
        $this->name = "$address->address_firstname $address->address_lastname";
        $this->address = new StripeAddress($address);
    }
}

class StripePaymentIntentData {
    public $shipping;
    public $description;
    public $setup_future_usage;

    public function __construct(object $address, object $order)
    {
        $this->shipping = new StripeShippingIntent($address);
        $this->description = JText::sprintf('ORDER_NUMBER').' : '. $order->order_id;
        $this->setup_future_usage = 'on_session';
    }
}

class plgHikashoppaymentStripecheckout extends hikashopPaymentPlugin
{
	public $accepted_currencies = [
        'AUD','BRL','CAD','EUR','GBP','JPY','USD','NZD','CHF','HKD','SGD','SEK',
        'DKK','PLN','NOK','HUF','CZK','MXN','MYR','PHP','TWD','THB','ILS','TRY',
        'RUB'
    ];

	public $multiple = true;
	public $name = 'stripecheckout';
	public $doc_form = 'stripecheckout';

	public $session_id;

	function __construct(&$subject, $config) {
		parent::__construct($subject, $config);
	}

	function onBeforeOrderCreate(&$order,&$do){
		if(parent::onBeforeOrderCreate($order, $do) === true)
			return true;

		if((empty($this->payment_params->email) || empty($this->payment_params->api_secret)) && $this->plugin_data->payment_id == $order->order_payment_id) {
			$this->app->enqueueMessage('Please check your &quot;PayPal&quot; plugin configuration');
			$do = false;
		}
	}

    private function storeStripeCustomerId(int $user_id, string $stripe_user_id)
    {
        $userClass = hikashop_get('class.user');
        $dbUser = $userClass->get($user_id);

        $user = new stdClass();
        $user->user_id = $dbUser->user_id;
        $user->user_params = $dbUser->user_params;
        $user->user_params->stripe_customer_id = $stripe_user_id;

        $userClass = hikashop_get('class.user');
        $userClass->save($user);
	}

	function onAfterOrderConfirm(&$order, &$methods, $method_id)
    {
		parent::onAfterOrderConfirm($order, $methods, $method_id);

		\Stripe\Stripe::setApiKey($this->payment_params->api_secret);

		// Get address info from HikaShop
        $address_type = $this->payment_params->address_type . '_address';


	    $stripeOrder = StripeOrder::create($order)
	        ->createCart()
		    ->addAdditionalItems()
	        ->addShippingCostConditional()
	        ->addPaymentCostConditional()
	        ->createPaymentIntent($address_type)
		    ->createStripeSession($this->url_itemid);

		$session = $stripeOrder->getSession();

		// Set the Session Id so we can redirect to it.
		$this->session_id = $session->id;

		// redirect to intermediate page which redirects to stripe checkout
		return $this->showPage('end');
	}

    /**
     * Webhook to set payment status
     *
     * @param $statuses
     *
     * @return bool|void
     */
    function onPaymentNotification(&$statuses)
    {
        $payload = @file_get_contents('php://input');

        \Stripe\Stripe::setApiKey($this->payment_params->api_secret);


        // we need the ID to load params
        $payloadDecoded = json_decode($payload);

        $dbOrder = $this->getOrder($payloadDecoded->data->object->client_reference_id);
        $this->loadPaymentParams($dbOrder);
        $this->loadOrderData($dbOrder);

        $endpoint_secret = $this->payment_params->endpoint_secret;

        // this ensures data is signed
        $sig_header = $_SERVER['HTTP_STRIPE_SIGNATURE'];
        $event = null;

        try {
            $event = \Stripe\Webhook::constructEvent(
                $payload, $sig_header, $endpoint_secret
            );
        } catch(\UnexpectedValueException $e) {
            // Invalid payload
            http_response_code(400);
            exit();
        } catch(\Stripe\Exception\SignatureVerificationException $e) {
            // Invalid signature
            http_response_code(400);
            exit();
        }

        // Handle the checkout.session.completed event
        if ($event->type == 'checkout.session.completed') {
            // Payment is completed so we only verify the order
            ['client_reference_id' => $order_id, 'customer' => $customer] = $event->data->object;

            if(empty($this->payment_params)) {
                return false;
            }

            if (! isset($this->user->user_params->stripe_customer_id) || $this->user->user_params->stripe_customer_id !== $customer) {
                $this->storeStripeCustomerId($dbOrder->order_user_id, $customer);
            }


            $order_status = $this->payment_params->verified_status;

            if ($dbOrder->order_status == $order_status) return true;

            $history = new stdClass();
            $history->notified = 0;
            $history->data = print_r($event, true);

            $config =& hikashop_config();
            if($config->get('order_confirmed_status', 'confirmed') == $order_status) {
                $history->notified = 1;
            }

            $email = new stdClass();
            $email->subject = JText::sprintf('PAYMENT_NOTIFICATION_FOR_ORDER','Paypal','CONFIRMED',$dbOrder->order_number);
            $email->body = str_replace('<br/>',"\r\n",JText::sprintf('PAYMENT_NOTIFICATION_STATUS','Paypal','CONFIRMED')).' '.JText::sprintf('ORDER_STATUS_CHANGED',$order_status)."\r\n\r\n".$order_text;

            $this->modifyOrder($order_id, $order_status, $history, $email);
        }

        http_response_code(200);

		return true;
	}

	function onPaymentConfiguration(&$element) {
		$subtask = JRequest::getCmd('subtask', '');
		if($subtask == 'ips') {
			$ips = null;
			echo implode(',', $this->_getIPList($ips));
			exit;
		}

		parent::onPaymentConfiguration($element);
		$this->address = hikashop_get('type.address');

		if(empty($element->payment_params->email)) {
			$app = JFactory::getApplication();
			$lang = JFactory::getLanguage();
			$locale = strtolower(substr($lang->get('tag'), 0, 2));
			$app->enqueueMessage(JText::sprintf('ENTER_INFO_REGISTER_IF_NEEDED', 'PayPal', JText::_('HIKA_EMAIL'), 'PayPal', 'https://www.paypal.com/' . $locale . '/mrb/pal=SXL9FKNKGAEM8'));
		}

        if(defined('OPENSSL_VERSION_NUMBER') && OPENSSL_VERSION_NUMBER < 0x009080bf ){
            $app = JFactory::getApplication();
            $app->enqueueMessage('The OpenSSL version installed on your server is too old and payment notifications will be rejected by PayPal. Please contact your hosting company in order to update it.');
        }
	}

	function onPaymentConfigurationSave(&$element) {
		if(!empty($element->payment_params->ips))
			$element->payment_params->ips = explode(',', $element->payment_params->ips);

		if(strpos($element->payment_params->url,'https://')===false){
			$app = JFactory::getApplication();
			$app->enqueueMessage('The URL must start with https://');
			return false;
		}

		return true;
	}

	function getPaymentDefaultValues(&$element) {
		$element->payment_name = 'Stripe';
		$element->payment_description='You can pay by credit card, apple pay or google pay using this method.';
		$element->payment_images = 'MasterCard,VISA,Credit_card';

		$element->payment_params->notification = 1;
		$element->payment_params->invalid_status = 'cancelled';
		$element->payment_params->pending_status = 'created';
		$element->payment_params->verified_status = 'confirmed';
		$element->payment_params->address_override = 1;
	}
}
