<?php
/**
 * @package    cookieconsent
 *
 * @author     christophf <your@email.com>
 * @copyright  A copyright
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 * @link       http://your.url.com
 */

use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Document\HtmlDocument;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Database\DatabaseDriver;

defined('_JEXEC') or die;

/**
 * CookieConsent plugin.
 *
 * @package   cookieconsent
 * @since     1.0.0
 */
class plgSystemCookieconsent extends CMSPlugin
{
	/**
	 * Application object
	 *
	 * @var    CMSApplication
	 * @since  1.0.0
	 */
	protected $app;

    /**
     * Domain where our app lives
     *
     * @var string
     * @since 1.0.0
     */
	protected $ckmnstrDomain = 'ckmnstr.de';

	/**
	 * Database object
	 *
	 * @var    DatabaseDriver
	 * @since  1.0.0
	 */
	protected $db;

    /**
     * Document Object
     *
     * @var HtmlDocument
     * @since 1.0.0
     */
	protected $doc;

	/**
	 * Affects constructor behavior. If true, language files will be loaded automatically.
	 *
	 * @var    boolean
	 * @since  1.0.0
	 */
	protected $autoloadLanguage = true;

	/**
	 * onAfterRender.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function onAfterDispatch()
	{
		// Access to plugin parameters
        if ($this->app->isClient('administrator')) return;

        $consents = $this->params->get('consents', []);
        $id = $this->params->get('id', null);
        $this->doc = $this->app->getDocument();

        $cdnDomain = "https://cdn.$this->ckmnstrDomain";

        $this->doc->addStyleSheet($cdnDomain . '/css/app.latest.css');
        $this->doc->addScript($cdnDomain . '/js/chunk-vendors.latest.js');
        $this->doc->addScript($cdnDomain . '/js/app.latest.js');

        $this->doc->addCustomTag("<script type='text/javascript'>
          document.addEventListener('DOMContentLoaded', function() {
            CookieConsent({
              basePath: '$cdnDomain/',
              appId: '$id',
            });
          });
</script>");

        foreach ($consents as $consent) {
            if (preg_match('/^https?:\/\//', $consent->script)) {
                $tag = $this->generateScriptTag($consent);
            } else {
                $tag = $this->generateScriptDeclaration($consent);
            }

            $this->addScriptTag($tag);
        }

        if ($this->params->get('handle_iframes', false)) {
            $this->doc->addStyleDeclaration('
            :root {
                --cc-accent-background: #194e9a;
            }
            .cc_banner {
                box-shadow: 0 0 7px rgba(0, 0, 0, 0.2), 0 -2px 10px rgba(0, 0, 0, 0.07);
            }
            .privacy-iframe-wrapper.privacy-blocked:before{
                content: "Video aufgrund der Datenschutzeinstellungen geblockt."; 
                position:absolute;
                width: 100%;
                top: 0;
                height: 100%;
                z-index: 1;
                display:flex;
                font-size:2em;
                justify-content:center;
                align-items:center;
                background: rgba(0,0,0,.5);
                color: rgb(255, 255, 255);
                text-shadow: 0 0 15px #000;
            }
            .privacy-iframe {
                background: none center/cover;
            }');

            $this->addScriptTag($this->generateScriptDeclaration((object) [
                'consent' => 'youtube',
                'script' => '[].slice.call(document.querySelectorAll(".vendor-youtube")).each(function(el){
                    el.src = el.dataset.src;
                    el.parentElement.classList.remove("privacy-blocked");
                });',
            ]));


            $this->addScriptTag($this->generateScriptDeclaration((object) [
                'consent' => 'youtube_privacy',
                'script' => '[].slice.call(document.querySelectorAll(".vendor-youtube")).each(function(el){
                    el.src = el.dataset.src;
                    el.parentElement.classList.remove("privacy-blocked");
                });',
            ]));
        }
	}

    protected function generateScriptDeclaration($consent): string
    {
        return "<script type='text/plain' data-type='text/javascript' data-consent='$consent->consent'>$consent->script</script>";
    }

    protected function generateScriptTag($consent): string
    {
        return "<script type='text/plain' data-type='text/javascript' data-consent='$consent->consent' data-src='$consent->script'></script>";
    }

    protected function addScriptTag(string $tag): void
    {
        $this->doc->addCustomTag($tag);
    }

    public function onAfterRender()
    {
        if ($this->app->isClient('administrator')) return;

        if (!$this->params->get('handle_iframes', false)) return;

        $content = $this->app->getBody();

        $dom = new DOMDocument;
        $dom->loadHTML($content);

        /** @var DOMNodeList $iframes */
        $iframes = $dom->getElementsByTagName('iframe');

        /** @var DOMElement $iframe */
        foreach ($iframes as $iframe) {
            $src = $this->noCookieUrl($iframe->getAttribute('src'));
            $wrapper = $iframe->parentNode;

            if (strpos($src, 'youtu') !== false ||
                strpos($src, 'vimeo') !== false
            ) {
                $iframe->setAttribute('data-src', $src);
                $vendorId = $this->getVendorId($src);

                $class = $iframe->getAttribute('class');

                $classList = array_filter([$class, 'privacy-iframe', $this->getIframeClass($vendorId)]);

                $iframe->setAttribute('class', implode(' ', $classList));
                $iframe->setAttribute('style',
                    'background-image: url(' . $this->getThumbnail($src, $vendorId) . ')');

                $iframe->removeAttribute('src');

                $wrapperClass = $wrapper->getAttribute('class');
                $wrapperClassList = $classList = array_filter([$wrapperClass, 'privacy-iframe-wrapper', 'privacy-blocked', $this->getIframeClass($vendorId) . '-wrapper']);
                $wrapper->setAttribute('class', implode(' ', $wrapperClassList));
            }
        }

        $this->app->setBody($dom->saveHTML());
    }

    protected function noCookieUrl(string $src): string
    {
        return preg_replace('/youtube\./', 'youtube-nocookie.', $src);
    }

    protected function getVendorId(string $src): ?int
    {
        if (strpos($src, 'youtu') !== false) {
            return 1;
        } else if (strpos($src, 'vimeo') !== false) {
           return 2;
        }
        return null;
    }

    protected function getIframeClass(int $vendorId): ?string
    {
        if ($vendorId === 1) {
            $vendor = 'youtube';
        } else if ($vendorId === 2) {
            $vendor = 'vimeo';
        } else {
            return null;
        }

        return "vendor-$vendor";
    }

    protected function getThumbnail(string $src, int $vendorId)
    {
        $videoId = $this->getVideoId($src, $vendorId);
        return "https://i.ytimg.com/vi/$videoId/maxresdefault.jpg";
    }

    protected function getVideoId(string $src, int $vendorId): ?string
    {
        $parts = parse_url($src);
        if (isset($parts['query'])) {
            parse_str($parts['query'], $query);
        } else {
            $query = null;
        }

        if ($vendorId === 1) {
            // Youtube
            return substr($parts['path'], 7);
        } else if ($vendorId === 2) {
            // vimeo
            return substr($parts['path'], 1);
        } else if ($vendorId === 3) {
            // Facebook
            if ($query) {
                return $query['v'];
            } else if (preg_match('/(?:videos\/|v=)([\d]*)/', $this->url, $matches)) {
                return $matches[1];
            }
        }

        return '';
    }
}
