<?php
/**
 * @package    cookieconsent
 *
 * @author     christophf <your@email.com>
 * @copyright  A copyright
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 * @link       http://your.url.com
 */

use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Cache\Cache;
use Joomla\CMS\Document\HtmlDocument;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Database\DatabaseDriver;

defined('_JEXEC') or die;

/**
 * CookieConsent plugin.
 *
 * @package   cookieconsent
 * @since     1.0.0
 */
class plgSystemCookieconsent extends CMSPlugin
{
	/**
	 * Application object
	 *
	 * @var    CMSApplication
	 * @since  1.0.0
	 */
	protected $app;

    /**
     * Domain where our app lives
     *
     * @var string
     * @since 1.0.0
     */
	protected $ckmnstrDomain = 'ckmnstr.de';

	/**
	 * Database object
	 *
	 * @var    DatabaseDriver
	 * @since  1.0.0
	 */
	protected $db;

    /**
     * Document Object
     *
     * @var HtmlDocument
     * @since 1.0.0
     */
	protected $doc;

	/**
	 * Affects constructor behavior. If true, language files will be loaded automatically.
	 *
	 * @var    boolean
	 * @since  1.0.0
	 */
	protected $autoloadLanguage = true;


    public function onAfterInitialise()
    {
        JLoader::registerNamespace('ckMonster', __DIR__ . '/lib');

        // Handle Joomla 4 Namespace changes
        if (!class_exists(ckMonster\Handlers\BaseHandler::class)) {
	        JLoader::registerNamespace('ckMonster', __DIR__ . '/lib/ckMonster');
        }
    }

	/**
	 * onAfterRender.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function onAfterDispatch()
	{
		// Access to plugin parameters
        if (!$this->checkGates()) return;

        $consents = $this->params->get('consents', []);
        $id = $this->params->get('id', null);

        $customIcon = $this->params->get('icon', null);

        $this->doc = $this->app->getDocument();

        $cdnDomain = "https://cdn.$this->ckmnstrDomain";
        $version = '2.0.1';

        try {
            $assets = $this->getLatestScripts();

            $this->doc->addStyleSheet($cdnDomain . '/' . $assets['app.css']);
            $this->doc->addScript($cdnDomain . '/' . $assets['chunk-vendors.js']);
            $this->doc->addScript($cdnDomain . '/' . $assets['app.js']);
        } catch (Exception $exception) {
            $this->doc->addStyleSheet($cdnDomain . '/css/app.latest.css', [
                'version' => $version,
            ]);
            $this->doc->addScript($cdnDomain . '/js/chunk-vendors.latest.js', [
                'version' => $version,
            ]);
            $this->doc->addScript($cdnDomain . '/js/app.latest.js', [
                'version' => $version,
            ]);
        }

        $options = [
            'basePath' => "$cdnDomain/",
            'appId' => $id,
            'btnPosition' => $this->params->get('btnPosition', 'bottom right'),
        ];

        if ($customIcon) {
            $options['icon'] = $customIcon;
        }

        if ($overrides = $this->params->get('overrides', null)) {
        	$options['overrides'] = json_decode($overrides);
        }

        $this->doc->addCustomTag("<script type=\"text/javascript\">
          document.addEventListener('DOMContentLoaded', function() {
            window.ckMnstr = CookieConsent(" . json_encode($options) . ");
          });
</script>");

        foreach ($consents as $consent) {
            if (preg_match('/^https?:\/\//', $consent->script)) {
                $tag = $this->generateScriptTag($consent);
            } else {
                $tag = $this->generateScriptDeclaration($consent);
            }

            $this->addScriptTag($tag);
        }

        if ($this->params->get('handle_iframes', false)) {
            $this->doc->addStyleDeclaration('
            div[data-blocked-consent] {
			    display: flex;
			    justify-content: center;
			    align-items: center;
			    background: rgba(0, 0, 0, 0.1);
			}
            .cc_banner {
                box-shadow: 0 0 7px rgba(0, 0, 0, 0.2), 0 -2px 10px rgba(0, 0, 0, 0.07);
            }
            .privacy-iframe {
                background: none center/cover;
            }');
        }
	}

    protected function generateScriptDeclaration($consent): string
    {
        return "<script type=\"text/plain\" data-type=\"text/javascript\" data-consent=\"$consent->consent\">$consent->script</script>";
    }

    protected function generateScriptTag($consent): string
    {
        return "<script type=\"text/plain\" data-type=\"text/javascript\" data-consent=\"$consent->consent\" data-src=\"$consent->script\"></script>";
    }

    protected function addScriptTag(string $tag): void
    {
        $this->doc->addCustomTag($tag);
    }

    public function onAfterRender()
    {
        if (!$this->params->get('handle_iframes', false)) return;

        if (!$this->checkGates()) return;

        $content = $this->app->getBody();

        if ($this->shouldHandleContent($content)) {
            $content = $this->injectCharset($content);

            // Save content including script content for later use.
            $rawContent = $content;

            // Replace scripts with md5 checksums so each gets replaced by its old value afterwards
            [$scripts, $checksums, $content] = $this->hashScripts($content);

            $dom = new DOMDocument;
            $dom->registerNodeClass(DOMElement::class, \ckMonster\Dom\ExtendedDOMElement::class);
            $dom->resolveExternals = true;
            $dom->substituteEntities = false;
            @$dom->loadHTML($content);

            if (\ckMonster\Handlers\IframeHandler::hasElements($content)) {
                \ckMonster\Handlers\IframeHandler::handle($dom, $this->app);
            }

	        if (\ckMonster\Handlers\FacebookFeedPro::hasElements($content)) {
		        \ckMonster\Handlers\FacebookFeedPro::handle($dom, $this->app);
	        }

            if (\ckMonster\Handlers\GoogleTranslate::hasElements($content)) {
                \ckMonster\Handlers\GoogleTranslate::handle($dom, $this->app);
            }

            if (\ckMonster\Handlers\GoogleMaps::hasElements($content)) {
                \ckMonster\Handlers\GoogleMaps::handle($dom, $this->app);

                if (\ckMonster\Handlers\GoogleMapsExtensions\SobiPro::hasElements($content)) {
                	\ckMonster\Handlers\GoogleMapsExtensions\SobiPro::handle($dom, $this->app, $rawContent);
                }

	            if (\ckMonster\Handlers\GoogleMapsExtensions\JEvents::hasElements($content)) {
		            \ckMonster\Handlers\GoogleMapsExtensions\JEvents::handle($dom, $this->app, $rawContent);
		            \ckMonster\Handlers\GoogleMapsExtensions\JEvents::modScripts($scripts);
	            }
            }

            if (\ckMonster\Handlers\ReCaptcha::hasElements($content)) {
                \ckMonster\Handlers\ReCaptcha::handle($dom, $this->app);

                if (\ckMonster\Handlers\ReCaptchaExtensions\PWebContactForm::hasElements($content)) {
                    \ckMonster\Handlers\ReCaptchaExtensions\PWebContactForm::handle($dom, $this->app, $rawContent);
                    \ckMonster\Handlers\ReCaptchaExtensions\PWebContactForm::modScripts($scripts);
                }

                if (\ckMonster\Handlers\ReCaptchaExtensions\RsForm::hasElements($content)) {
                    \ckMonster\Handlers\ReCaptchaExtensions\RsForm::handle($dom, $this->app, $rawContent);
                }

                if (\ckMonster\Handlers\ReCaptchaExtensions\Acymailing::hasElements($content)) {
                    \ckMonster\Handlers\ReCaptchaExtensions\Acymailing::handle($dom, $this->app, $rawContent);
                }
            }

            if (\ckMonster\Handlers\ReCaptchaExtensions\HikaShop::hasElements($content)) {
                \ckMonster\Handlers\ReCaptchaExtensions\HikaShop::handle($dom, $this->app, $rawContent);
                \ckMonster\Handlers\ReCaptchaExtensions\HikaShop::modScripts($scripts);
            }

            if (\ckMonster\Handlers\JuxInstagram::hasElements($content)) {
                \ckMonster\Handlers\JuxInstagram::handle($dom, $this->app, $rawContent);
                \ckMonster\Handlers\JuxInstagram::modScripts($scripts);
            }

            $content = $dom->saveHTML();

            // Reset script content
            $content = str_replace($checksums, $scripts, $content);

            $content = preg_replace_callback('#<(\w+)([^>]*)\s*/>#s', function($matches){

                // ignore only these tags
                $xhtml_tags = array('br', 'hr', 'input', 'frame', 'img', 'area', 'link', 'col', 'base', 'basefont', 'param' ,'meta');

                // if a element that is not in the above list is empty,
                // it should close like   `<element></element>` (for eg. empty `<title>`)
                return in_array($matches[1], $xhtml_tags) ? "<{$matches[1]}{$matches[2]} />" : "<{$matches[1]}{$matches[2]}></{$matches[1]}>";
            }, $content);

            $this->app->setBody($content);
        }
    }

    public function injectCharset($content)
    {
        if (!$GLOBALS['charset_injected'])
        {
            if (!preg_match('/<meta http-equiv="Content-Type" content="text\/html; charset=utf-8"\s?\/?>/i', $content))
            {
                $content = preg_replace('/<meta charset="utf-8"\s?\/?>/', '<meta charset="utf-8" /><meta http-equiv="Content-Type" content="text/html; charset=utf-8" />', $content, 1);
            }
            $GLOBALS['charset_injected'] = true;
        }

        return $content;
    }

    public function shouldHandleContent($content): bool
    {
        return
            \ckMonster\Handlers\JuxInstagram::hasElements($content) ||
            \ckMonster\Handlers\GoogleTranslate::hasElements($content) ||
            \ckMonster\Handlers\FacebookFeedPro::hasElements($content) ||
            \ckMonster\Handlers\ReCaptcha::hasElements($content) ||
            \ckMonster\Handlers\IframeHandler::hasElements($content) ||
            \ckMonster\Handlers\GoogleMaps::hasElements($content);
    }

    public function hashScripts($content): array
    {
        preg_match_all("/(?:(<script[^>]*>)(.*?))<\/script>/s", $content, $scripts, PREG_SET_ORDER);

        $checksums = [];
        $checksumsContent = [];
        $scriptTags = [];
        $scriptContents = [];

        foreach ($scripts as $script) {
            [$fullmatch, $scriptTag, $scriptContent] = $script;

            if (empty($scriptContent)) continue;

            $checksum = '/** ckMnstr_hash: ' . md5($scriptTag . $scriptContent) . ' **/';

            // Replace scriptTags with checksumTag
            $scriptTags[] = $fullmatch;
            $checksums[] = "{$scriptTag}{$checksum}</script>";

            // For reverse, we replace the checkSumContent with the script content.
            // Otherwise, we can't modify script tags.
            $checksumsContent[] = $checksum;
            $scriptContents[] = $scriptContent;
        }

        $content   = str_replace($scriptTags, $checksums, $content);

        return array($scriptContents, $checksumsContent, $content);
    }

    protected function checkGates(): bool
    {
        if ($this->app->isClient('administrator')) return false;

        if ($this->disabledUsersId()) return false;

        if ($this->isEditPage()) {
        	return (bool) ($this->params->get('enable_edit_forms', false));
        }

        if (!$this->isHTMLView()) return false;

        return true;
    }

    protected function isHTMLView()
    {
        return ($this->app->getDocument()->getType() === 'html');
    }

	protected function disabledUsersId()
	{
		$users = $this->params->get('disabled_users', []);
		$currentUser = JFactory::getUser();

		return ($currentUser->id && in_array($currentUser->id, $users));
    }

    protected function isEditPage(): bool
    {
        $input = $this->app->input;

        $option = $input->get('option');

        // always return false for these components
        if (in_array($option, ['com_rsevents', 'com_rseventspro']))
        {
            return false;
        }

        $task = $input->get('task');

        if (strpos($task, '.') !== false)
        {
            $task = explode('.', $task);
            $task = array_pop($task);
        }

        $view = $input->get('view');

        if (strpos($view, '.') !== false)
        {
            $view = explode('.', $view);
            $view = array_pop($view);
        }

        return
            (
                in_array($option, ['com_contentsubmit', 'com_cckjseblod'])
                || ($option == 'com_comprofiler' && in_array($task, ['', 'userdetails']))
                || in_array($task, ['edit', 'form', 'submission'])
                || in_array($view, ['edit', 'form'])
                || in_array($input->get('do'), ['edit', 'form'])
                || in_array($input->get('layout'), ['edit', 'form', 'write'])
            );
    }

    protected static function cachedCallback(string $id, callable $callback, Cache $cache)
    {
        $cached = $cache->get($id);

        if (empty($cached)) {
            $value = $callback();
            $cache->store(serialize($value), $id);
            return $value;
        }

        return unserialize($cached);
    }

    /**
     * @return array
     * @throws Exception
     */
    private function getLatestScripts(): array
    {
        if (!extension_loaded('curl')) {
            throw new Exception('Curl not loaded');
        }

        $cache = new Cache([
            'defaultgroup' => 'plg_cookieconsent',
            'caching' => true,
            'lifetime' => 60 * 60 *24,
        ]);

        return static::cachedCallback('manifest', function () {
            return $this->getManifest();
        }, $cache);
    }

    private function getManifest(): array
    {
        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => "https://cdn.$this->ckmnstrDomain/manifest.json",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 5,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "GET",
            CURLOPT_POSTFIELDS => "",
        ));

        $response = curl_exec($curl);
        $err = curl_error($curl);
        $errNo = curl_errno($curl);

        curl_close($curl);

        if ($err) {
            throw new Exception($err, $errNo);
        }
        return json_decode($response, true);
    }
}
