<?php
/**
 * @package    cookieconsent
 *
 * @author     christophf <your@email.com>
 * @copyright  A copyright
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 * @link       http://your.url.com
 */

use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\Document\HtmlDocument;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Database\DatabaseDriver;

defined('_JEXEC') or die;

/**
 * CookieConsent plugin.
 *
 * @package   cookieconsent
 * @since     1.0.0
 */
class plgSystemCookieconsent extends CMSPlugin
{
	/**
	 * Application object
	 *
	 * @var    CMSApplication
	 * @since  1.0.0
	 */
	protected $app;

    /**
     * Domain where our app lives
     *
     * @var string
     * @since 1.0.0
     */
	protected $ckmnstrDomain = 'ckmnstr.de';

	/**
	 * Database object
	 *
	 * @var    DatabaseDriver
	 * @since  1.0.0
	 */
	protected $db;

    /**
     * Document Object
     *
     * @var HtmlDocument
     * @since 1.0.0
     */
	protected $doc;

	/**
	 * Affects constructor behavior. If true, language files will be loaded automatically.
	 *
	 * @var    boolean
	 * @since  1.0.0
	 */
	protected $autoloadLanguage = true;


    public function onAfterInitialise()
    {
        JLoader::registerNamespace('ckMonster', __DIR__ . '/lib');
    }

	/**
	 * onAfterRender.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function onAfterDispatch()
	{
		// Access to plugin parameters
        if (!$this->checkGates()) return;

        $consents = $this->params->get('consents', []);
        $id = $this->params->get('id', null);

        $customIcon = $this->params->get('icon', null);

        $this->doc = $this->app->getDocument();

        $cdnDomain = "https://cdn.$this->ckmnstrDomain";
        $version = '2.0.1';

        $this->doc->addStyleSheet($cdnDomain . '/css/app.latest.css', [
            'version' => $version,
        ]);
        $this->doc->addScript($cdnDomain . '/js/chunk-vendors.latest.js', [
            'version' => $version,
        ]);
        $this->doc->addScript($cdnDomain . '/js/app.latest.js', [
            'version' => $version,
        ]);

        $options = [
            'basePath' => "$cdnDomain/",
            'appId' => $id,
        ];

        if ($customIcon) {
            $options['icon'] = $customIcon;
        }

        $this->doc->addCustomTag("<script type='text/javascript'>
          document.addEventListener('DOMContentLoaded', function() {
            CookieConsent(" . json_encode($options) . ");
          });
</script>");

        foreach ($consents as $consent) {
            if (preg_match('/^https?:\/\//', $consent->script)) {
                $tag = $this->generateScriptTag($consent);
            } else {
                $tag = $this->generateScriptDeclaration($consent);
            }

            $this->addScriptTag($tag);
        }

        if ($this->params->get('handle_iframes', false)) {
            $this->doc->addStyleDeclaration('
            .cc_banner {
                box-shadow: 0 0 7px rgba(0, 0, 0, 0.2), 0 -2px 10px rgba(0, 0, 0, 0.07);
            }
            .privacy-iframe {
                background: none center/cover;
            }');
        }
	}

    protected function generateScriptDeclaration($consent): string
    {
        return "<script type='text/plain' data-type='text/javascript' data-consent='$consent->consent'>$consent->script</script>";
    }

    protected function generateScriptTag($consent): string
    {
        return "<script type='text/plain' data-type='text/javascript' data-consent='$consent->consent' data-src='$consent->script'></script>";
    }

    protected function addScriptTag(string $tag): void
    {
        $this->doc->addCustomTag($tag);
    }

    public function onAfterRender()
    {
        if (!$this->params->get('handle_iframes', false)) return;

        if (!$this->checkGates()) return;

        $content = $this->app->getBody();

        if ($this->shouldHandleContent($content)) {
            $content = $this->injectCharset($content);

            // Replace scripts with md5 checksums so each gets replaced by its old value afterwards
            [$scripts, $checksums, $content] = $this->hashScripts($content);

            $dom = new DOMDocument;
            $dom->resolveExternals = true;
            $dom->substituteEntities = false;
            @$dom->loadHTML($content);

            if (\ckMonster\Handlers\IframeHandler::hasElements($content)) {
                \ckMonster\Handlers\IframeHandler::handle($dom, $this->app);
            }

            if (\ckMonster\Handlers\SobiGoogle::hasElements($content)) {
                \ckMonster\Handlers\SobiGoogle::handle($dom, $this->app);
            }

            $content = $dom->saveHTML();

            // Reset script content
            $content = str_replace($checksums, $scripts[3], $content);

            $content = preg_replace_callback('#<(\w+)([^>]*)\s*/>#s', function($matches){

                // ignore only these tags
                $xhtml_tags = array('br', 'hr', 'input', 'frame', 'img', 'area', 'link', 'col', 'base', 'basefont', 'param' ,'meta');

                // if a element that is not in the above list is empty,
                // it should close like   `<element></element>` (for eg. empty `<title>`)
                return in_array($matches[1], $xhtml_tags) ? "<{$matches[1]}{$matches[2]} />" : "<{$matches[1]}{$matches[2]}></{$matches[1]}>";
            }, $content);

            $this->app->setBody($content);
        }
    }

    public function injectCharset($content)
    {
        if (!$GLOBALS['charset_injected'])
        {
            if (!preg_match('/<meta http-equiv="Content-Type" content="text\/html; charset=utf-8"\s?\/?>/i', $content))
            {
                $content = preg_replace('/<meta charset="utf-8"\s?\/?>/', '<meta charset="utf-8" /><meta http-equiv="Content-Type" content="text/html; charset=utf-8" />', $content, 1);
            }
            $GLOBALS['charset_injected'] = true;
        }

        return $content;
    }

    public function shouldHandleContent($content): bool
    {
        return \ckMonster\Handlers\IframeHandler::hasElements($content) || \ckMonster\Handlers\SobiGoogle::hasElements($content);
    }

    public function hashScripts($content): array
    {
        preg_match_all("/((<script.*>)(.*))<\/script>/sU", $content, $scripts);
        $checksums = array_map('md5', $scripts[3]);
        $content   = str_replace($scripts[3], $checksums, $content);

        return array($scripts, $checksums, $content);
    }

    protected function checkGates(): bool
    {
        if ($this->app->isClient('administrator')) return false;

        if ($this->isEditPage()) return false;

        return true;
    }

    protected function isEditPage(): bool
    {
        $input = $this->app->input;

        $option = $input->get('option');

        // always return false for these components
        if (in_array($option, ['com_rsevents', 'com_rseventspro']))
        {
            return false;
        }

        $task = $input->get('task');

        if (strpos($task, '.') !== false)
        {
            $task = explode('.', $task);
            $task = array_pop($task);
        }

        $view = $input->get('view');

        if (strpos($view, '.') !== false)
        {
            $view = explode('.', $view);
            $view = array_pop($view);
        }

        return
            (
                in_array($option, ['com_contentsubmit', 'com_cckjseblod'])
                || ($option == 'com_comprofiler' && in_array($task, ['', 'userdetails']))
                || in_array($task, ['edit', 'form', 'submission'])
                || in_array($view, ['edit', 'form'])
                || in_array($input->get('do'), ['edit', 'form'])
                || in_array($input->get('layout'), ['edit', 'form', 'write'])
            );
    }
}
