<?php


namespace ckMonster\Handlers;

use ckMonster\CookieVendors;
use ckMonster\HandlerInterface;
use Joomla\CMS\Cache\Cache;
use function foo\func;

class IframeHandler extends BaseHandler implements HandlerInterface
{
    public function getElements(): \DOMNodeList
    {
        return $this->dom->getElementsByTagName('iframe');
    }

    public static function hasElements(string $content): bool
    {
        return strpos($content, '<iframe') !== false;
    }

    public function handleElement(\DOMElement $iframe): void
    {
        $src = $this->noCookieUrl($iframe->getAttribute('src'));

        if ($vendorId = $this->getVendorId($src))
        {
            $iframe->setAttribute('data-src', $src);

            $class = $iframe->getAttribute('class');
            $width = (int) $iframe->getAttribute('width') ?: 1280;

            $classList = array_filter([$class, 'privacy-iframe']);

            $iframe->setAttribute('data-consent', $this->getVendorName($vendorId));

            $iframe->setAttribute('class', implode(' ', $classList));

            // Check if iframe is a video
            if (in_array($vendorId, CookieVendors::VIDEO_VENDORS)) {
                $thumb = $this->cachedCallback($src, function () use ($src, $vendorId, $width) {
                    return $this->getThumbnail($src, $vendorId, $width);
                });

                if ($thumb) {
                    $iframe->setAttribute('style',
                        'background-image: url(' . $thumb . ')');
                }
            }

            $iframe->removeAttribute('src');
        }
    }

    protected function noCookieUrl(string $src): string
    {
        return preg_replace('/youtube\./', 'youtube-nocookie.', $src);
    }

    protected function getVendorId(string $src): ?int
    {
        if (strpos($src, 'youtu') !== false) {
            return CookieVendors::YOUTUBE;
        } else if (strpos($src, 'vimeo') !== false) {
            return CookieVendors::VIMEO;
        } else if (strpos($src, 'technikboerse') !== false) {
            return CookieVendors::TECHNIKBOERSE;
        } else if (strpos($src, 'google.com/maps/embed') !== false) {
            return CookieVendors::GOOGLEMAPS;
        }
        return null;
    }

    protected function getVendorName(int $vendorId): ?string
    {
        if ($vendorId === CookieVendors::YOUTUBE) {
            return 'youtube';
        } else if ($vendorId === CookieVendors::VIMEO) {
            return 'vimeo';
        } else if ($vendorId === CookieVendors::TECHNIKBOERSE) {
            return 'technikboerse';
        } else if ($vendorId === CookieVendors::GOOGLEMAPS) {
            return 'google-maps';
        }

        return null;
    }

    protected function getThumbnail(string $src, int $vendorId, int $width): ?string
    {
        $videoId = $this->getVideoId($src, $vendorId);

        if ($vendorId === CookieVendors::YOUTUBE) {
            return "https://i.ytimg.com/vi/$videoId/maxresdefault.jpg";
        } else if ($vendorId === CookieVendors::VIMEO) {
            $OEmbed = $this->getOembed($src, $vendorId, $width);

            return $OEmbed->thumbnail_url;
        }

        return null;
    }

    protected function getVideoId(string $src, int $vendorId): ?string
    {
        $parts = parse_url($src);
        if (isset($parts['query'])) {
            parse_str($parts['query'], $query);
        } else {
            $query = null;
        }

        if ($vendorId === CookieVendors::YOUTUBE) {
            // Youtube
            return substr($parts['path'], 7);
        } else if ($vendorId === CookieVendors::VIMEO) {
            // vimeo
            return substr($parts['path'], 1);
        } else if ($vendorId === CookieVendors::FACEBOOK) {
            // Facebook
            if ($query) {
                return $query['v'];
            } else if (preg_match('/(?:videos\/|v=)([\d]*)/', $this->url, $matches)) {
                return $matches[1];
            }
        }

        return '';
    }

    private function getOembed(string $src, int $vendorId, int $width): ?object
    {
        $videoId = $this->getVideoId($src, $vendorId);

        $parameters = [
            'width' => $width,
        ];

        if ($vendorId === CookieVendors::VIMEO) {
            $parameters['url'] = "https://vimeo.com/$videoId";
            $str = $this->cURL('https://vimeo.com/api/oembed.json', $parameters);
        }

        return ($str) ? json_decode($str) : null;
    }

    protected function cURL(string $url, array $parameters = []): string
    {
        $query = http_build_query($parameters);

        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => "$url?$query",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 3,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "GET",
            CURLOPT_POSTFIELDS => "",
            CURLOPT_HTTPHEADER => array(
                "Accept: application/json",
                "Referer: " . \JUri::root(),
            ),
        ));

        $response = curl_exec($curl);
        $err = curl_error($curl);
        $errNo = curl_errno($curl);

        curl_close($curl);

        if ($err) {
            throw new Exception($err, $errNo);
        }
        return $response;
    }
}
