<?php

use Joomla\CMS\Cache\Cache;
use Joomla\CMS\Helper\ModuleHelper;

if (!class_exists('MenusHelper')) {
	JLoader::register('MenusHelper', JPATH_ADMINISTRATOR . '/components/com_menus/helpers/menus.php');
}

class Softgarden
{
	public string $jobList;
	public array $categories;
	public array $audiences;
	public array $exclude;
	protected $version;
	protected Cache $cache;
	protected $doc;

	public function __construct(\Joomla\Registry\Registry $params)
	{
		$this->doc = JFactory::getDocument();

		$this->jobList = $params->get('jobList');
		$this->categories = $this->getFilteredParams($params, 'categories');
		$this->audiences = $this->getFilteredParams($params, 'audiences');
		$this->exclude = $this->getFilteredParams($params, 'exclude');

		$this->cache = new Cache([
			'defaultgroup' => 'mod_softgarden',
			'caching' => true,
			'lifetime' => 3600,
		]);

		$this->appendMedia();
	}

	protected function getFilteredParams(\Joomla\Registry\Registry $params, string $param, string $default = ''): array
	{
		return array_filter(array_map('trim', explode(',', $params->get($param, $default))));
	}

	protected function cacheKey(): string
	{
		return md5(join('_', [
			join('.', $this->categories),
			$this->jobList,
		]));
	}

	public function getData(bool $grouped = false, $direction = 'DESC')
	{
	    $jobs = $this->prepareData();

	    if ($grouped) {
        $groupedArr = array_reduce($jobs, function (array $accumulator, object $element) {
          $accumulator[$element->group][] = $element;

          return $accumulator;
        }, []);

        uasort($groupedArr, function($a, $b) {
          return $a['group'] <=> $b['group'];
        });

        if ($direction === 'ASC') {
          return array_reverse($groupedArr, true);
        }

        return $groupedArr;
      }

		return $jobs;
	}

	protected function getCatalog(string $catalogKey): object
	{
		return $this->cachedCallback($catalogKey, function() use ($catalogKey) {
			return $this->_fetch("https://api.softgarden.io/api/rest/v3/frontend/catalogs/$catalogKey?locale=DE");
		});
	}

	protected function getJobs(): array
	{
		return array_filter($this->cachedCallback($this->cacheKey(), function () {
			return $this->_fetch("https://api.softgarden.io/api/rest/v3/frontend/jobslist/$this->jobList")->results;
		}), function ($jobEntry) {
			if (!empty($this->exclude) && in_array((string) $jobEntry->jobDbId, $this->exclude)) {
				return false;
			}
			if (!empty($this->categories) && !array_intersect($jobEntry->jobCategories, $this->categories)) {
				return false;
			}
			if (!empty($this->audiences) && !array_intersect($jobEntry->audiences, $this->audiences)) {
				return false;
			}
			return true;
		});
	}

	protected function prepareData(): array
	{
		$jobs = $this->getJobs();
		$categories = $this->getCatalog('JOB_CATEGORY');
		$audiences = $this->getCatalog('AUDIENCE');

		return array_map(function($job) use ($categories, $audiences) {
			$job->audiences = array_map(fn($aud) => $audiences->{$aud}, $job->audiences);
			$job->jobCategories = array_map(fn($cat) => $categories->{$cat}, $job->jobCategories);

			$job->group = preg_match('/2\d{3}(?:-(?:12|11|0[0-9]))?/', $job->keywords, $matches) ? $matches[0] : null;

			return $job;
		}, $jobs);
	}

	protected function cachedCallback(string $id, callable $callback)
	{
		$cached = $this->cache->get($id);

		if (empty($cached)) {
			$value = $callback();
			$this->cache->store(serialize($value), $id);
			return $value;
		}

		return unserialize($cached);
	}


	protected function _fetch(string $url)
	{
		$curl = curl_init();

		curl_setopt_array($curl, [
			CURLOPT_URL => $url,
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING => "",
			CURLOPT_MAXREDIRS => 10,
			CURLOPT_TIMEOUT => 30,
			CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
			CURLOPT_CUSTOMREQUEST => "GET",
			CURLOPT_POSTFIELDS => "",
			CURLOPT_HTTPHEADER => [
				"Authorization: Basic YTVmMmE5YmEtMWRkNi00OGQ1LTg0MmEtYmZlNDI5NjRmMmVkOg==",
				"Content-Type: application/json"
			],
		]);

		$response = curl_exec($curl);
		$err = curl_error($curl);

		curl_close($curl);

		if ($err) {
			throw new Exception($err);
		} else {
			return json_decode($response);
		}
	}

	protected function appendMedia()
	{
		$this->doc->addStyleSheet(JUri::base(true) . '/media/mod_softgarden/css/softgarden.css', [
			'version' => $this->hashedVersion(),
		]);
	}

	protected function hashedVersion(): string
	{
		return substr(sha1($this->getVersion()), 0, 8);
	}

	protected function getVersion(): string
	{
		if ($this->version) return $this->version;

		$xml = simplexml_load_file(__DIR__ . "/mod_softgarden.xml");

		return $this->version = (string) $xml->version;
	}
}
