<?php

use Joomla\CMS\Helper\ModuleHelper;

class SiemaConfig
{
    public $selector;
    public $perPage = 5;
    public $rows = 1;
    public $indicator = true;
    public $navigation = true;
    public $loop = false;

    protected $cast = [
        'perPage' => 'int',
        'rows' => 'int',
        'indicator' => 'bool',
        'navigation' => 'bool',
        'loop' => 'bool',
        'breakpoints' => 'object',
    ];

    public function __construct(\Joomla\Registry\Registry $params)
    {
        $properties = get_object_vars($this);
        foreach($properties as $prop => $value) {
            $this->{$prop} = $this->castAttribute($prop, $params->get($prop, $value));
        }
    }

    private function castAttribute(string $prop, $value)
    {
        if (isset($this->cast[$prop]) && $propType = $this->cast[$prop]) {
            switch ($propType) {
                case 'object':
                    return (object) $value;
                case 'integer':
                case 'int':
                    return (int) $value;
                case 'boolean':
                case 'bool':
                    return (bool) $value;
            }
        }
        return $value;
    }
}

class SiemaItemsHelper extends ArrayObject
{
    public function offsetSet($name, $value)
    {
        if (!($value instanceof SiemaItem))
        {
            throw new InvalidArgumentException(sprintf('Only objects of type SiemaItem allowed.'));
        }
        parent::offsetSet($name, $value);
    }

    public function __toString()
    {
        return json_encode(array_values($this->getArrayCopy()));
    }
    
    public function __toArray()
    {
        
    }
}

class SiemaItems extends SiemaItemsHelper
{
    public function __construct(string $itemsString)
    {
        $items = [];
        $itemsArray = json_decode($itemsString, true);
        $elementCount = count(reset($itemsArray));

        for ($i = 0; $i < $elementCount; $i++) {
            $items[] = new SiemaItem(array_combine(array_keys($itemsArray), array_column($itemsArray, $i)));
        }

        parent::__construct($items);
    }
}

class SiemaArticles extends SiemaItemsHelper
{
    public function __construct(int $catId, $introLength = 200)
    {
        parent::__construct(self::getItems($catId, $introLength));
    }

    private function getItems(int $catId, ?int $introLength): array
    {
        $items = JFactory::getDBO()->setQuery("SELECT * FROM #__content WHERE catid = $catId")->loadObjectList();
        return array_map(function($item) use ($introLength) {
            return self::prepareItem($item, $introLength);
        }, $items);
    }

    private static function prepareItem($item, ?int $introLength)
    {
        // echo '<pre>' . print_r([$item], true). '</pre>';
        return new SiemaItem([
            'title' => $item->title,
            'image' => self::getIntroImage($item),
            'description' => self::getIntro($item->introtext, $introLength),
        ]);
    }

    private static function getIntro(string $value, ?int $limit = 200, string $allowedTags = '<p><i><u><a><b><ul><li><strong>'): string
    {
        if ($allowedTags !== false) {
            $value = strip_tags($value, $allowedTags);
        }

        if ($limit === null) {
            return $value;
        }

        $intro = preg_replace( "/\r|\n/", "", self::substrHtml($value, $limit));

        if (strlen($intro) < strlen($value)) {
            $intro = preg_replace('/<\/p>$/', '…</p>', $intro);
        }

        return $intro;
    }

    private static function substrHtml(string $value, int $limit = 200): string
    {
        if (mb_strwidth($value, 'UTF-8') <= $limit) {
            return $value;
        }

        // Strip text with HTML tags, sum html len tags too.
        // Is there another way to do it?
        do {
            $len          = mb_strwidth($value, 'UTF-8');
            $len_stripped = mb_strwidth(strip_tags($value), 'UTF-8');
            $len_tags     = $len - $len_stripped;

            $value = mb_strimwidth($value, 0, $limit + $len_tags, '', 'UTF-8');
        } while ($len_stripped > $limit);

        // Load as HTML ignoring errors
        $dom = new DOMDocument();
        @$dom->loadHTML('<?xml encoding="utf-8" ?>'.$value, LIBXML_HTML_NODEFDTD);

        // Fix the html errors
        $value = $dom->saveHtml($dom->getElementsByTagName('body')->item(0));

        // Remove body tag
        $value = mb_strimwidth($value, 6, mb_strwidth($value, 'UTF-8') - 13, '', 'UTF-8'); // <body> and </body>
        // Remove empty tags
        return preg_replace('/<(\w+)\b(?:\s+[\w\-.:]+(?:\s*=\s*(?:"[^"]*"|"[^"]*"|[\w\-.:]+))?)*\s*\/?>\s*<\/\1\s*>/', '', $value);

    }

    private static function getIntroImage(object $article): ?string
    {
        $images = json_decode($article->images);

        if ($images->image_intro) return $images->image_intro;

        return null;
    }
}

class SiemaItem
{
    public $title;
    public $image;
    public $description;

    public function __construct(array $params)
    {
        $properties = get_object_vars($this);
        foreach($properties as $prop => $value) {
            $this->{$prop}($params[$prop] ?? $value);
        }
    }

    private function setImage($value)
    {
        return $this->image = "/$value";
    }

    public function __call($name, $arguments)
    {
        $prefix = count($arguments) === 1 ? 'set' : 'get';

        $methodName = "$prefix$name";

        if (method_exists(get_class($this), $methodName))
        {
            if ($prefix === 'get') return $this->{$methodName}();
            return $this->{$methodName}($arguments[0]);
        }

        if (property_exists(get_class($this), $name)) {
            if ($prefix === 'set') $this->$name = $arguments[0];
            return $this->$name;
        }

        throw new \Exception("Property $name doesn't exist.");
    }
}

class Siema
{
    static public $NextID = 1;
    protected $id;
    protected $doc;
    protected $baseUrl;
    protected $assetsUrl;
    protected $entryPoint = 'main';
    protected $manifest = [];
    protected $config;

    public function __construct(SiemaConfig $config)
    {
        $config->selector = $this->id = 'carousel-' . self::$NextID++;
        $this->doc = JFactory::getDocument();
        $this->baseUrl = JUri::root(true);
        $this->assetsUrl = "$this->baseUrl/media/mod_carousel/assets";
        $this->config = $config;
        $this->manifest = json_decode(file_get_contents(JPATH_ROOT . '/media/mod_carousel/manifest.json'), true);
    }

    private function attachScripts()
    {
        $this->addScript($this->manifest["$this->entryPoint.js"] );
    }

    private function attachStyles()
    {
        $this->addStyleSheet($this->manifest["$this->entryPoint.css"]);
    }

    private function addScript(string $script)
    {
        $this->doc->addScript("$this->assetsUrl/$script", [
            'defer' => true,
            'async' => true,
        ]);
    }

    private function createStartScript()
    {
        $config = json_encode($this->config);


        $html[] = '<script>';
        $html[] = "new PxSlider($config);";
        $html[] = '</script>';

        return implode('', $html);
    }

    private function addStyleSheet(string $style)
    {
        $this->doc->addStyleSheet("$this->assetsUrl/$style");
    }

    public function render($items, $moduleclass_sfx, $layout = 'default', $rows = 1)
    {
        $this->attachScripts();
        $this->attachStyles();

        $carouselId = $this->id;

        $this->config->selector = "#$carouselId";

        require ModuleHelper::getLayoutPath('mod_carousel', $layout);

        echo $this->createStartScript();
    }
}
