<?php

use Joomla\CMS\Helper\ModuleHelper;

if (!class_exists('MenusHelper')) {
	JLoader::register('MenusHelper', JPATH_ADMINISTRATOR . '/components/com_menus/helpers/menus.php');
}

class SiemaObject
{
    protected $cast = [];

    public function __construct(\Joomla\Registry\Registry $params)
    {
        $properties = get_object_vars($this);
        foreach($properties as $prop => $value) {
            $this->{$prop} = $this->castAttribute($prop, $params->get($prop, $value));
        }
    }

    protected function castAttribute(string $prop, $value)
    {
        if (isset($this->cast[$prop]) && $propType = $this->cast[$prop]) {
            switch ($propType) {
                case 'SiemaBreakpoints':
                    return (new SiemaBreakpoints($value))->getArrayCopy();
                case 'array':
                    return (array) array_values(get_object_vars($value));
                case 'object':
                    return (object) $value;
                case 'int|string':
                    return is_numeric($value) ? (int) $value : $value;
                case 'integer':
                case 'int':
                    return (int) $value;
                case 'boolean':
                case 'bool':
                    return (bool) $value;
            }
        }
        return $value;
    }
}


class SiemaBreakpoint extends SiemaObject
{
    public $maxWidth;
    public $perPage;

    protected $cast = [
        'perPage' => 'int',
        'maxWidth' => 'int|string',
    ];

    public function __construct(object $params)
    {
        $properties = get_object_vars($this);
        foreach($properties as $prop => $value) {
            $this->{$prop} = $this->castAttribute($prop, $params->{$prop} ?? $value);
        }
    }
}

class SiemaBreakpoints extends ArrayObject
{
    public function offsetSet($name, $value)
    {
        if (!($value instanceof SiemaBreakpoint))
        {
            throw new InvalidArgumentException(sprintf('Only objects of type SiemaBreakpoint allowed.'));
        }
        parent::offsetSet($name, $value);
    }

    public function __construct($items)
    {
        if (is_object($items)) {
            $items = array_values(get_object_vars($items));
        }

        $items = array_map(function($item) {
            return new SiemaBreakpoint($item);
        }, $items);

        if ($items) {
            parent::__construct($items);
        }
    }

    public function __toString()
    {
        return json_encode(array_values($this->getArrayCopy()));
    }
}

class SiemaConfig extends SiemaObject
{
    public $selector;
    public $perPage = 5;
    public $rows = 1;
    public $indicator = true;
    public $navigation = true;
    public $loop = false;
    public $showPageNumber = false;
    public $showTitle = false;
    public $indicatorBefore = false;
    public $useImageAsBackground = false;
    public $duration = 250;
    public $breakpoints;
    public $autoplay;
    public $autoplayDuration;
    public $easing = 'ease-in-out';

    protected $cast = [
        'perPage' => 'int',
        'rows' => 'int',
        'showPageNumber' => 'bool',
        'showTitle' => 'bool',
        'useImageAsBackground' => 'bool',
        'indicator' => 'bool',
        'navigation' => 'bool',
        'loop' => 'bool',
        'breakpoints' => 'SiemaBreakpoints',
        'indicatorBefore' => 'bool',
        'autoplay' => 'bool',
        'autoplayDuration' => 'int',
        'duration' => 'int',
    ];

    public function __toString()
    {
        return json_encode($this);
    }
}

class SiemaItemsHelper extends ArrayObject
{
    public function offsetSet($name, $value)
    {
        if (!($value instanceof SiemaItem))
        {
            throw new InvalidArgumentException(sprintf('Only objects of type SiemaItem allowed.'));
        }
        parent::offsetSet($name, $value);
    }

    public function __toString()
    {
        return json_encode(array_values($this->getArrayCopy()));
    }

	private function updateIndexes() {
		$this->exchangeArray(array_values($this->getArrayCopy()));
	}

	public function usort(callable $callback): void
	{
		$this->uasort($callback);
		$this->updateIndexes();
	}
}

class SiemaItems extends SiemaItemsHelper
{
    public function __construct($items)
    {
        if (is_object($items)) {
            $items = array_values(get_object_vars($items));
        }

        $items = array_map(function($item) {
            return new SiemaItem($item);
        }, $items);

        parent::__construct($items);
    }
}

class SiemaArticles extends SiemaItemsHelper
{
    private $langTag;

    public function __construct(int $catId, $introLength = 200, ?int $limit = 0, bool $activeOnly = true, ?string $sortKey = null, ?string $sortOrder = 'asc')
    {
        $this->langTag = JFactory::getLanguage()->getTag();
        parent::__construct(self::getItems($catId, $introLength, $limit, $activeOnly, $sortKey, $sortOrder));
    }

    private function getItems(int $catId, ?int $introLength, ?int $limit, bool $activeOnly, ?string $sortKey, ?string $sortOrder = 'asc'): array
    {
    	$db = JFactory::getDBO();
	    $query = $db->getQuery(true);

	    $conditions = [
	    	"catid = $catId",
		    "language IN ('*', '$this->langTag')"
	    ];

	    $query
		    ->select(['*'])
		    ->where($conditions)
		    ->from('#__content');

	    if ($activeOnly) {
	    	$query
			    ->andWhere('state = 1')
			    ->andWhere([
				    'publish_up IS NULL',
				    'publish_up <= NOW()',
			    ], 'OR')
			    ->andWhere([
				    'publish_down IS NULL',
				    'publish_down >= NOW()',
			    ], 'OR');
	    }

	    // if sorted by field don't limit query here
	    if ($sortKey && strpos($sortKey, 'field') === false) {
            $query->order("$sortKey " . strtoupper($sortOrder));
            if ($limit > 0) {
                $query->setLimit($limit);
            }
        }

        $items = JFactory::getDBO()->setQuery($query)->loadObjectList();

        $items = array_map(function($item) use ($introLength, $catId) {
            return $this->prepareItem($item, $catId, $introLength);
        }, $items);


        if ($sortKey && strpos($sortKey, 'field') !== false) {
            usort($items, function(SiemaItem $a, SiemaItem $b) use ($sortKey, $sortOrder) {
                if ($sortOrder === 'desc') return $b->get($sortKey) <=> $a->get($sortKey);
                return $a->get($sortKey) <=> $b->get($sortKey);
            });
        }

	    // slicing array here again
        return array_slice($items, 0, $limit);
    }

    private function prepareItem($item, int $catId, ?int $introLength): SiemaItem
    {
        // echo '<pre>' . print_r([$item], true). '</pre>';
	    $fields = FieldsHelper::getFields('com_content.article', $item, true);
        return new SiemaItem((object) array_merge((array) $item, [
            'image' => self::getIntroImage($item),
            'description' => self::getIntro($item->introtext, $introLength),
            'link' => $this->getLink($item, $catId),
	        'fields' => array_column($fields, null, 'name'),
        ]));
    }

    private function getLink(object $item, int $catId)
    {
        // $associations = MenusHelper::getAssociations($ItemId);
        // if ($associations) {
        //     $lang = JFactory::getLanguage()->getTag();
        //     $ItemId = $associations[$lang] ?? $ItemId;
        // }

        $query = http_build_query([
            'option' => 'com_content',
            'view' => 'article',
            'id' => "$item->id:$item->alias",
            'catid' => $catId,
            'lang' => $this->langTag,
        ]);
        return JRoute::_("index.php?$query");
    }

    private static function getIntro(string $value, ?int $limit = 200, string $allowedTags = '<p><i><u><a><b><ul><li><strong>'): string
    {
        // Remove Styles and Scripts from Intro
        $value = preg_replace("/<(?:style|script)[^>]*>[^<]*<\/\s*(?:style|script)>/i", '', $value);

        if ($allowedTags !== false) {
            $value = strip_tags($value, $allowedTags);
        }

        if ($limit === null) {
            return $value;
        }

        $intro = preg_replace( "/\r|\n/", "", self::substrHtml($value, $limit));

        if (strlen($intro) < strlen($value)) {
            $intro = preg_replace('/<\/p>$/', '…</p>', $intro);
        }

        return $intro;
    }

    private static function substrHtml(string $value, int $limit = 200): string
    {
        if (mb_strwidth($value, 'UTF-8') <= $limit) {
            return $value;
        }

        // Strip text with HTML tags, sum html len tags too.
        // Is there another way to do it?
        do {
            $len          = mb_strwidth($value, 'UTF-8');
            $len_stripped = mb_strwidth(strip_tags($value), 'UTF-8');
            $len_tags     = $len - $len_stripped;

            $value = mb_strimwidth($value, 0, $limit + $len_tags, '', 'UTF-8');
        } while ($len_stripped > $limit);

        // Load as HTML ignoring errors
        $dom = new DOMDocument();
        @$dom->loadHTML('<?xml encoding="utf-8" ?>'.$value, LIBXML_HTML_NODEFDTD);

        // Fix the html errors
        $value = $dom->saveHtml($dom->getElementsByTagName('body')->item(0));

        // Remove body tag
        $value = mb_strimwidth($value, 6, mb_strwidth($value, 'UTF-8') - 13, '', 'UTF-8'); // <body> and </body>
        // Remove empty tags
        return preg_replace('/<(\w+)\b(?:\s+[\w\-.:]+(?:\s*=\s*(?:"[^"]*"|"[^"]*"|[\w\-.:]+))?)*\s*\/?>\s*<\/\1\s*>/', '', $value);

    }

    private static function getIntroImage(object $article): ?string
    {
        $images = json_decode($article->images);

        if ($images->image_intro) return $images->image_intro;

        return null;
    }
}

class SiemaItem
{
    public $title;
    public $image;
    public $description;
    public $link;
    public $fields;
    public $created;
    public $publish_up;
    public $publish_down;

    public function __construct(object $params)
    {
        $properties = get_object_vars($this);
        foreach($properties as $prop => $value) {
            $this->{$prop}($params->{$prop} ?? $value);
        }
    }

    private function setImage($value)
    {
        return $this->image = "/$value";
    }

    public function __call($name, $arguments)
    {
        $prefix = count($arguments) === 1 ? 'set' : 'get';

        $methodName = "$prefix$name";

        if (method_exists(get_class($this), $methodName))
        {
            if ($prefix === 'get') return $this->{$methodName}();
            return $this->{$methodName}($arguments[0]);
        }

        if (property_exists(get_class($this), $name)) {
            if ($prefix === 'set') $this->$name = $arguments[0];
            return $this->$name;
        }

        throw new \Exception("Property $name doesn't exist.");
    }

	public function get($key, $default = null)
	{
		$params = explode('.', $key);
		if($params == 0) {
			return $this;
		} else {
			$obj = $this;
			foreach($params as $index => $param) {
				$obj = $obj->{$param} ?: $obj[$param] ?: $default;
			}
		}
		return $obj;
	}
}

class Siema
{
    static public $NextID = 1;
    protected $id;
    protected $doc;
    protected $baseUrl;
    protected $assetsUrl;
    protected $entryPoint = 'main';
    protected $manifest = [];
    protected $config;

    public function __construct(SiemaConfig $config)
    {
        $config->selector = $this->id = 'carousel-' . self::$NextID++;
        $this->doc = JFactory::getDocument();
        $this->baseUrl = JUri::root(true);
        $this->assetsUrl = "$this->baseUrl/media/mod_carousel/assets";
        $this->config = $config;
        $this->manifest = json_decode(file_get_contents(JPATH_ROOT . '/media/mod_carousel/manifest.json'), true);
    }

    private function attachScripts()
    {
        $this->addScript($this->manifest["$this->entryPoint.js"] );
    }

    private function attachStyles()
    {
        $this->addStyleSheet($this->manifest["$this->entryPoint.css"]);
    }

    private function addScript(string $script)
    {
        $this->doc->addScript("$this->assetsUrl/$script", [
            'defer' => true,
            'async' => true,
        ]);
    }

    private function createStartScript()
    {
        $config = (string) $this->config;

        $html[] = '<script>';
        $html[] = "new PxSlider($config);";
        $html[] = '</script>';

        return implode('', $html);
    }

    private function addStyleSheet(string $style)
    {
        $this->doc->addStyleSheet("$this->assetsUrl/$style");
    }

    public function render(SiemaItemsHelper $items, $moduleclass_sfx, $layout = 'default', $rows = 1)
    {
        $this->attachScripts();
        $this->attachStyles();

        $carouselId = $this->id;

        $this->config->selector = "#$carouselId";

        require ModuleHelper::getLayoutPath('mod_carousel', $layout);

        echo $this->createStartScript();
    }
}
