<?php
/**
 * @package    pixelcms_articles
 *
 * @author     christophf <your@email.com>
 * @copyright  A copyright
 * @license    GNU General Public License version 2 or later; see LICENSE.txt
 * @link       http://your.url.com
 */

use Joomla\Registry\Registry;

defined('_JEXEC') or die;


require_once 'BaseModel.php';

enum EnvironmentalBadge: int {
    case NONE = 1;
    case RED = 2;
    case YELLOW = 3;
    case GREEN = 4;
    case BLUE = 5;

    public function getTranslationKey()
    {
        return match ($this) {
            self::RED => 'red',
            self::YELLOW => 'yellow',
            self::GREEN => 'green',
            self::BLUE => 'blue',
            default => null,
        };
    }
}

enum FuelType: int {
    case Other = 0;
    case Petrol = 1; // Benzin
    case Diesel = 2;
    case LPG = 3; // Autogas
    case CNG = 4; // Compressed Natural Gas (Erdgas)
    case Electric = 6;
    case Hybrid = 7;
    case Hydrogen = 8; // Wasserstoff
    case Ethanol = 9;
    case HybridDiesel = 10;
    case BiFuel = 11;

    public function getTranslationKey()
    {
        return match ($this) {
            FuelType::Petrol => 'petrol',
            FuelType::Diesel => 'diesel',
            FuelType::LPG => 'lpg',
            FuelType::CNG => 'cng',
            FuelType::Electric => 'electric',
            FuelType::Hybrid => 'hybrid',
            FuelType::Hydrogen => 'hydrogen',
            FuelType::Ethanol => 'ethanol',
            FuelType::HybridDiesel => 'hybriddiesel',
            default => 'other',
        };
    }
}

enum GearType: int {
    case Unknown = 0;
    case Manual = 1;
    case SemiAutomatic = 2;
    case Automatic = 3;

    public function getTranslationKey()
    {
        return match ($this) {
            self::Unknown => 'unknown',
            self::Manual => 'manual',
            self::SemiAutomatic => 'semi_automatic',
            self::Automatic => 'automatic',
        };
    }
}

enum InteriorColor: int {
    case Black = 1;
    case Gray = 2;
    case Beige = 3;
    case Brown = 4;
    case Other = 5;
    case White = 6;
    case Blue = 7;
    case Red = 8;
    case Yellow = 9;
    case Green = 10;
    case Orange = 11;

    public function getTranslationKey()
    {
        return match ($this) {
            InteriorColor::Black => 'black',
            InteriorColor::Gray => 'gray',
            InteriorColor::Beige => 'beige',
            InteriorColor::Brown => 'brown',
            InteriorColor::White => 'white',
            InteriorColor::Blue => 'blue',
            InteriorColor::Red => 'red',
            InteriorColor::Yellow => 'yellow',
            InteriorColor::Green => 'green',
            InteriorColor::Orange => 'orange',
            default => 'other',
        };
    }
}

class BaseColor {
    static public function getTranslationKey(string $color)
    {
        $color = str_replace(['hell', 'dunkel'], '', $color);

        return match ($color) {
            'schwarz' => 'black',
            'grau' => 'gray',
            'beige' => 'beige',
            'braun' => 'brown',
            'weiß', 'weiss' => 'white',
            'blau' => 'blue',
            'rot' => 'red',
            'gelb' => 'yellow',
            'gruen', 'grün' => 'green',
            'orange' => 'orange',
            'silber' => 'silver',
            'violette', 'violett' => 'violet',
            default => ucfirst($color),
        };
    }
}

enum InteriorType: int {
    case Leather = 1;
    case PartialLeather = 2; // Teilleder
    case Fabric = 3; // Stoff
    case Velour = 4;
    case Alcantara = 5;
    case Other = 6;
    case SyntheticLeather = 7; // Kunstleder

    public function getTranslationKey(): string {
        return match ($this) {
            InteriorType::Leather => 'leather',
            InteriorType::PartialLeather => 'partial_leather',
            InteriorType::Fabric => 'fabric',
            InteriorType::Velour => 'velour',
            InteriorType::Alcantara => 'alcantara',
            InteriorType::SyntheticLeather => 'synthetic_leather',
            default => 'other',
        };
    }
}

enum AirbagOption: int {
    case DriverAirbag = 2; // Fahrer Airbag
    case FrontAirbags = 3; // Vordere Airbags
    case FrontAndSideAirbags = 4; // Vorder und Seiten Airbags
    case FrontSideAndMoreAirbags = 5; // Vorder und Seiten und weitere Airbags

    function getTranslationKey(): string {
        return match ($this) {
            AirbagOption::DriverAirbag => 'driver_airbag',
            AirbagOption::FrontAirbags => 'front_airbags',
            AirbagOption::FrontAndSideAirbags => 'front_and_side_airbags',
            AirbagOption::FrontSideAndMoreAirbags => 'front_side_and_more_airbags',
        };
    }
}

class Category {
    public float | int $id;
    public string $name;

    public function  __construct(array $data) {
        $cat = $data[VehicleData::Category->value];

        $this->id = hexdec(substr(md5($cat), 1, 3));
        $this->name = $cat;
    }
}

class Manufacturer {
    public float | int $id;
    public string $value;
    public ?string $sub_brand;

    public function  __construct(array $data) {
        $brand = $data[VehicleData::Brand->value];

        $vin = $data[VehicleData::Vin->value];
        if ($vin) {
            if (str_starts_with($vin, 'WV1') || str_starts_with($vin, 'WV2')) {
                $this->sub_brand = 'VW Nutzfahrzeuge';
            }
        }

        $this->value = (string) $brand;
        $this->id = hexdec(substr(sha1($brand), 0, 4));
    }
}

class Condition {
    public string $id = 'used';

    public function  __construct(array $data) {
        $this->id = (function($d) {
            switch (true) {
                case $d[VehicleData::NewVehicle->value]:
                    return 'new';
                case $d[VehicleData::ShowroomCar->value]:
                    return 'showroom_car';
                case $d[VehicleData::DemoVehicle->value]:
                    return 'demonstration_car';
                case $d[VehicleData::AccidentVehicle->value]:
                case $d[VehicleData::DamagedVehicle->value]:
                    return 'damaged';
                default:
                    return 'used';
            }
        })($data);
    }
}

class Climate {
    public ?string $id = 'used';

    static public function getValue(array $data) {
        return (function($d) {
            return match (true) {
                $d[VehicleData::FiveZoneClimateControl->value] => '5_zone_ac',
                $d[VehicleData::QuadrupleZoneClimateControl->value] => '4_zone_ac',
                $d[VehicleData::TripleZoneClimateControl->value] => '3_zone_ac',
                $d[VehicleData::DualZoneClimateControl->value] => '2_zone_ac',
                $d[VehicleData::ClimateControl->value] => '1_zone_ac',
                default => null,
            };
        })($data);
    }
}

class Mileage {
    public string $unit = 'Km';
    public ?float $value = null;
    public ?string $show = null;

    public function  __construct(array $data) {
        $kilometers = (float) $data[VehicleData::Kilometers->value];
        if ($data[VehicleData::Kilometers->value]) {
            $this->value = $kilometers;
            $this->show = number_format($kilometers, 0, ',', '.') . " $this->unit";
        }
    }
}

class Vat {
    public float | null $value;
    public bool $separable = true;

    public function  __construct(array $data) {
        $this->value = floatval(VehicleData::VatRate->convertValue($data[VehicleData::VatRate->value]));
        $this->separable = (bool) $data[VehicleData::Vat->value] == '0';
    }
}

class Power {
    public float $PS;
    public float $kW;

    public function  __construct(array $data) {
        $this->kW = (float) $data[VehicleData::Power->value];
        $this->PS = round($this->kW * 1.36);
    }
}

class Dealer {
    public int $id;
    public string $name;

    public function  __construct(array $data) {
        $dealer = $data[VehicleData::CustomerNumberOrBranch->value];

        $this->id = hexdec(substr(md5($dealer), 1, 3));
        $this->name = $dealer;
    }
}

// TODO: Check if this all is correct
class Price {
    public string $currency = '€';
    public bool $specialprice = false;
    public object $values;

    public function __construct(array $data)
    {
        $vat = floatval(VehicleData::VatRate->convertValue($data[VehicleData::VatRate->value]) ?? 0);
        $isBrutto = $vat && $data[VehicleData::Vat->value] == '1';

        $price = parse_float($data[VehicleData::Price->value]);
        $price_dealer = (float) $data[VehicleData::DealerPrice->value];

        // Only products with prices get calculated
        if ($isBrutto) {
            $this->values = (object) [
                'brutto' => $price,
                'netto' => round($price / (1 + $vat)),
                'dealer_price' => $price_dealer,
            ];
        } else {
            $this->values = (object) [
                'brutto' => round($price * (1 + $vat)),
                'netto' => $price,
                'dealer_price' => $price_dealer,
            ];
        }
    }
}

/**
 * Mapping for Mobile.de Export
 *
 * @see https://www.eautoseller.de/doku/mobilede_csv_api_ext.html
 *
 * @since 5.4.0
 */
enum VehicleData: int {
    case CustomerNumberOrBranch = 0; // kundennummer/filiale
    case InternalId = 1; // interne_nummer
    case Category = 2; // kategorie
    case Brand = 3; // marke
    case Model = 4; // modell
    case Power = 5; // leistung
    case MainInspection = 6; // hu
    case FirstRegistration = 8; // ez
    case Kilometers = 9; // kilometer
    case Price = 10; // preis
    case Vat = 11; // mwst
    case Oldtimer = 13; // oldtimer
    case Vin = 14; // vin
    case DamagedVehicle = 15; // beschädigtes_fahrzeug
    case Color = 16; // farbe
    case ClimateControl = 17; // klima
    case Taxi = 18; // taxi
    case DisabledAccessible = 19; // behindertengerecht
    case ShowroomCar = 20; // jahreswagen
    case NewVehicle = 21; // neufahrzeug
    case OurRecommendation = 22; // unsere empfehlung
    case DealerPrice = 23; // haendlerpreis
    case Remark = 25; // bemerkung
    case ImageId = 26; // bild_id
    case Metallic = 27; // metallic
    case Currency = 28; // waehrung
    /**
     * mwstsatz
     *
     * MwSt-Satz. Ganz- oder Fließkommazahl.
     * @see https://services.mobile.de/refdata/sites/GERMANY/vatrates
     */
    case VatRate = 29; // mwstsatz
    case Warranty = 30; // garantie
    case AlloyWheels = 31; // leichtmetallfelgen
    case Esp = 32; // esp
    case Abs = 33; // abs
    case TrailerHitch = 34; // anhaengerkupplung
    case Immobilizer = 36; // wegfahrsperre
    case NavigationSystem = 37; // navigationssystem
    case Sunroof = 38; // schiebedach
    case CentralLocking = 39; // zentralverriegelung
    case PowerWindows = 40; // fensterheber
    case FourWheelDrive = 41; // allradantrieb
    case Doors = 42; // tueren
    case EnvironmentalBadge = 43; // umweltplakette
    case PowerSteering = 44; // servolenkung
    case BiodieselSuitable = 45; // biodiesel
    case ServiceBook = 46; // scheckheftgepflegt
    case CatalyticConverter = 47; // katalysator
    case Kickstarter = 48; // kickstarter
    case EStarter = 49; // estarter
    case DemoVehicle = 50; // vorfuehrfahrzeug
    case Drive = 51; // antrieb
    case Ccm = 52; // ccm
    case LoadCapacity = 53; // tragkraft
    case Payload = 54; // nutzlast
    case TotalWeight = 55; // gesamtgewicht
    case LiftHeight = 56; // hubhoehe
    case ConstructionHeight = 57; // bauhoehe
    case YearOfConstruction = 58; // baujahr
    case OperatingHours = 59; // betriebsstunden
    case Seats = 60; // sitze
    case EmissionClass = 61; // schadstoff
    case CabinType = 62; // kabinenart
    case Axles = 63; // achsen
    case CruiseControl = 64; // tempomat
    case AuxiliaryHeating = 65; // standheizung
    case Cabin = 66; // kabine
    case ProtectiveRoof = 67; // schutzdach
    case FullFairing = 68; // vollverkleidung
    case Municipal = 69; // komunal
    case Crane = 70; // kran
    case RetarderIntarder = 71; // retarder_intarder
    case SleepingPlace = 72; // schlafplatz
    case Tv = 73; // tv
    case Wc = 74; // wc
    case TailLift = 75; // ladebordwand
    case HydraulicSystem = 76; // hydraulikanlage
    case SlidingDoorCommercial = 77; // schiebetuer nur Nutzfahrzeuge
    case AxleFormula = 78; // radformel
    case Partition = 79; // trennwand
    case Ebs = 80; // ebs
    case Rentable = 81; // vermietbar
    case Compressor = 82; // kompressor
    case AirSuspensionCommercial = 83; // luftfederung
    case DiscBrake = 84; // scheibenbremse
    case FrontHydraulics = 85; // fronthydraulik
    case Bss = 86; // bss
    case QuickChange = 87; // schnellwechsel
    case Zsa = 88; // zsa
    case Kitchen = 89; // kueche
    case CoolingBox = 90; // kuehlbox
    case SleepingSeats = 91; // schlafsitze
    case FrontLift = 92; // frontheber
    case VisibleOnlyForDealers = 93; // sichtbar_nur_fuer_Haendler
    case Envkv = 95; // envkv
    case ConsumptionCity = 96; // verbrauch_innerorts
    case ConsumptionHighway = 97; // verbrauch_ausserorts
    case ConsumptionCombined = 98; // verbrauch_kombiniert
    case Emission = 99; // emission
    case XenonHeadlights = 100; // xenonscheinwerfer
    case SeatHeating = 101; // sitzheizung
    case ParticulateFilter = 102; // partikelfilter
    case ParkingAssistance = 103; // einparkhilfe
    case SchwackeCode = 104; // schwackecode
    case SupplyDate = 105; // lieferdatum (Nor quite sure whats the difference between this and the auslieferungsdatum
    case DeliveryPeriod = 106; // lieferfrist
    case TransportCosts = 107; // ueberfuehrungskosten
    case HuAuNew = 108; // hu/au_neu
    case FuelType = 109; // kraftstoffart
    case GearType = 110; // getriebeart
    case ExportVehicle = 111; // exportfahrzeug
    case DayRegistration = 112; // tageszulassung
    case EyeCatcher = 113; // blickfaenger
    case Hsn = 114; // hsn
    case Tsn = 115; // tsn
    case Page1Ad = 116; // seite_1_inserat
    case E10Compatible = 119; // e10
    case QualitySeal = 120; // qualitaetssiegel
    case VegetableOil = 121; // pflanzenoel
    case Scr = 122; // scr
    case Case = 123; // koffer
    case CrashBar = 124; // sturzbuegel
    case Windshield = 125; // scheibe
    case StandAirConditioning = 126; // standklima
    case SsTire = 127; // s-s-bereifung
    case RoadRegistration = 128; // strassenzulassung
    case BunkBed = 129; // etagenbett
    case FixedBed = 130; // festbett
    case RearGarage = 131; // heckgarage
    case Awning = 132; // markise
    case SeparateShower = 133; // sep-dusche
    case SolarSystem = 134; // solaranlage
    case MiddleSeatGroup = 135; // mittelsitzgruppe
    case RoundSeatGroup = 136; // rundsitzgruppe
    case SideSeatGroup = 137; // seitensitzgruppe
    case HailDamage = 138; // hagelschaden
    case SleepingPlaces = 139; // schlafplaetze
    case VehicleLength = 140; // fahrzeuglaenge
    case VehicleWidth = 141; // fahrzeugbreite
    case VehicleHeight = 142; // fahrzeughoehe
    case LoadSpacePallet = 143; // laderaum-europalette
    case LoadVolume = 144; // laderaum-volumen
    case LoadLength = 145; // laderaum-laenge
    case LoadWidth = 146; // laderaum-breite
    case LoadHeight = 147; // laderaum-hoehe
    case ListingAsNew = 148; // inserat als 'neu' markieren
    case EffectiveAnnualRate = 149; // effektiver jahreszins
    case MonthlyRate = 150; // monatliche rate
    case Duration = 151; // laufzeit
    case DownPayment = 152; // anzahlung
    case FinalInstallment = 153; // schlussrate
    case FinancingFeature = 154; // finanzierungsfeature
    case InteriorColor = 155; // interieurfarbe
    case InteriorType = 156; // interieurtyp
    case Airbag = 157; // airbag
    case PreviousOwners = 158; // vorbesitzer
    case TopAd = 159; // top inserat
    case GrossCreditAmount = 160; // bruttokreditbetrag
    case ClosingFees = 161; // abschlussgebuehren
    case RateProtection = 162; // ratenabsicherung
    case NetCreditAmount = 163; // nettokreditbetrag
    case ProviderBank = 164; // anbieterbank
    case NominalInterestRate = 165; // soll-zinssatz
    case TypeOfNominalInterestRate = 166; // art des soll-zinssatzes
    case CountryVersion = 167; // landesversion
    case VideoUrl = 168; // video-url
    case EnergyEfficiencyClass = 169; // energieeffizienzklasse
    case EnvkvFuelType = 170; // envkv_benzin_sorte
    case ElectricMirrors = 171; // elektrische seitenspiegel
    case SportsSuspension = 172; // sportfahrwerk
    case SportsPackage = 173; // sportpaket
    case Bluetooth = 174; // bluetooth
    case OnBoardComputer = 175; // bordcomputer
    case CdPlayer = 176; // cd spieler
    case ElectricSeatAdjustment = 177; // elektrische sitzeinstellung
    case HeadUpDisplay = 178; // head-up display
    case HandsFreeSystem = 179; // freisprecheinrichtung
    case Mp3Interface = 180; // mp3 schnittstelle
    case MultifunctionSteeringWheel = 181; // multifunktionslenkrad
    case SkiBag = 182; // skisack
    case TunerOrRadio = 183; // tuner oder radio
    case SportsSeats = 184; // sportsitze
    case PanoramicRoof = 185; // panorama dach
    case ChildSeatAttachment = 186; // kindersitzbefestigung
    case CorneringLight = 187; // kurvenlicht
    case LightSensor = 188; // lichtsensor
    case FogLights = 189; // nebelscheinwerfer
    case DaytimeRunningLights = 190; // tagfahrlicht
    case TractionControl = 191; // traktionskontrolle
    case StartStopAutomatic = 192; // start stop automatik
    case RainSensor = 193; // regensensor
    case NonSmokerVehicle = 194; // nichtraucher fahrzeug
    case RoofRailing = 195; // dachreling
    case AccidentVehicle = 196; // unfallfahrzeug
    case Roadworthy = 197; // fahrtauglich
    case ProductionDate = 198; // produktionsdatum
    case ParkingSensorsFront = 199; // einparkhilfe sensoren vorne
    case ParkingSensorsRear = 200; // einparkhilfe sensoren hinten
    case ParkingCamera = 201; // einparkhilfe kamera
    case ParkingAssistanceSelfSteering = 202; // einparkhilfe selbstlenkendes system
    case RotstiftPrice = 204; // rotstiftpreis
    case EbayExport = 205; // ebayk export
    case PluginHybrid = 206; // plugin hybrid
    case CombinedElectricConsumption = 207; // kombinierter stromverbrauch
    case Highlight1 = 208; // highlight 1
    case Highlight2 = 209; // highlight 2
    case Highlight3 = 210; // highlight 3
    case FinancingConditions = 211; // bedingungen finanzierungsvorschlag
    case HeadlightCleaningSystem = 212; // scheinwerferreinigungsanlage
    case HighBeamAssistant = 213; // fernlichtassistent
    case AmbientLighting = 214; // ambientes licht
    case RearSeatHeating = 215; // sitzheizung hinten
    case RearElectricSeatAdjustment = 216; // elektrische sitzeinstellung hinten
    case SmokerPackage = 217; // raucherpaket
    case FoldablePassengerSeat = 218; // umklappbarer beifahrersitz
    case FullDigitalCombinationInstrument = 219; // volldigitales kombiinstrument
    case WinterPackage = 220; // winterpaket
    case AndroidAuto = 221; // android auto
    case AppleCarplay = 222; // apple carplay
    case InductionChargingSmartphones = 223; // induktionsladen smartphones
    case IntegratedMusicStreaming = 224; // musikstreaming integriert
    case Wifi = 225; // w-lan
    case AllSeasonTires = 226; // reifen-allwetterreifen
    case SummerTires = 227; // reifen-sommerreifen
    case WinterTires = 228; // reifen-winterreifen
    case SpareWheel = 229; // notrad
    case PunctureKit = 230; // pannenkitt
    case FullSizeSpareWheel = 231; // reserverad
    case SteelWheel = 232; // stahlfelge
    case MissingEVBattery = 233; // e-fahrzeug-batterie-fehlt
    case BoughtEVBattery = 234; // e-fahrzeug-batterie-gekauft
    case RentedEVBattery = 235; // e-fahrzeug-batterie-gemietet
    case RangeExtender = 236; // range extender
    case IsofixPassenger = 237; // isofix beifahrer
    case LuggageCompartmentSeparation = 238; // gepäckraumabtrennung
    case SwivelTowBar = 239; // anhängerkupplung schwenkbar
    case RemovableTowBar = 240; // anhängerkupplung abnehmbar
    case FixedTowBar = 241; // anhängerkupplung starr
    case SlidingDoorRight = 242; // schiebetür rechts
    case SlidingDoorLeft = 243; // schiebetür links
    case SlidingDoorsBothSides = 244; // schiebetür beidseitig
    case BiXenonLight = 245; // bi-xenon-licht
    case LaserHeadlights = 246; // laserlicht-scheinwerfer
    case AlarmSystem = 247; // alarmanlage
    case HeatedSteeringWheel = 248; // beheizbares lenkrad
    case AutomaticallyDimmingInteriorMirror = 249; // innenspiegel automatisch abbl.
    case LaneKeepingAssistant = 250; // spurhalteassistent
    case AdaptiveCruiseControl = 251; // abstandstempomat
    case LedHeadlights = 252; // led-scheinwerfer
    case EmergencyBrakeAssistant = 253; // notbremsassistent
    case ProximityWarning = 254; // abstandswarner
    case LaneChangeAssistant = 255; // spurwechselassistent
    case SeatVentilation = 256; // sitzbelüftung
    case CenterArmrest = 257; // mittelarmlehne
    case HeatedWindshield = 258; // beheizbare frontscheibe
    case HillStartAssist = 259; // berganfahrassistent
    case DabRadio = 260; // dab-radio
    case ElectricTailgate = 261; // elektrische heckklappe
    case LedDaytimeRunningLight = 262; // led-tagfahrlicht
    case LeatherSteeringWheel = 264; // lederlenkrad
    case LumbarSupport = 265; // lordosenstütze
    case AirSuspension = 266; // luftfederung
    case MassageSeat = 267; // massagesitz
    case FatigueWarningSystem = 268; // müdigkeitswarnsystem
    case NightVisionAssistant = 269; // nachtsicht-assistent
    case EmergencyCallSystem = 270; // notrufsystem
    case TirePressureMonitoringSystem = 271; // reifendruckkontrollsystem
    case PaddleShifters = 272; // schaltwippen
    case SlidingDoor = 273; // schiebetür
    case SoundSystem = 274; // soundsystem
    case VoiceControl = 275; // sprachsteuerung
    case Touchscreen = 277; // touchscreen
    case Usb = 278; // usb
    case TrafficSignRecognition = 279; // verkehrszeichenerkennung
    case KeylessCentralLocking = 280; // schlüssellose zentralsverriegelung
    case DualZoneClimateControl = 281; // klimaautomatik 2 zonen
    case TripleZoneClimateControl = 282; // klimaautomatik 3 zonen
    case QuadrupleZoneClimateControl = 283; // klimaautomatik 4 zonen
    case Camera360ParkingAssistance = 284; // einparkhilfe kamera 360
    case NonGlaringHighBeam = 285; // blendfreies fernlicht (matrix)
    case CompanyCar = 286; // dienstwagen
    case FormerRecommendedRetailPrice = 287; // ehem. preisempfehlung uvp
    case EquipmentLine = 288; // ausstattungslinie
    case ConstructionType = 289; // bauart
    case AcceptanceDate = 290; // hereinnahmedatum
    case MobileDeTurboAd = 291; // mobile.de Turboinserat
    case EffectiveAnnualRateLeasing = 292; // effektiver jahreszins
    case MonthlyRateLeasing = 293; // monatliche rate
    case DurationLeasing = 294; // laufzeit
    case DownPaymentLeasing = 295; // anzahlung
    case FinalInstallmentLeasing = 296; // schlussrate
    case TargetGroupLeasing = 297; // zielgruppe
    case TotalLeasingMileage = 298; // laufleistung
    case LeasingResidualValue = 299; // leasing restwert
    case LeasingExtraMileageCost = 300; // leasing mehrkilometer
    case LeasingReducedMileageCost = 301; // leasing minderkilometer
    case ProviderBankLeasing = 302; // anbieterbank
    case NominalInterestRateLeasing = 303; // soll-zinssatz
    case TypeOfNominalInterestRateLeasing = 304; // art des soll-zinssatzes
    case ConditionsLeasing = 305; // bedingungen leasingvorschlag
    case TotalLeasingAmount = 306; // leasinggesamtbetrag
    case NetCreditAmountLeasing = 307; // nettokreditbetrag
    case DetailedEmissionClass = 308; // schadstoffklassedetail
    case SpeedLimitationDevice = 309; // geschwindigkeitsbegrenzungsanlage
    case PurchasePrice = 310; // ekpreis
    case RefurbishmentCosts = 311; // instandsetzungskosten
    case MobileDeClicks = 312; // mobile.de klicks
    case Co2EmissionWltp = 313; // co2 emission wltp
    case OpenLeads = 314; // leads offen
    case OpenLeadsHandler = 315; // leads offen bearbeiter
    case LatestOpenLeadDate = 316; // leads offen aktuell
    case StorageLocation = 317; // Lagerplatz
    case ModelBase = 318; // Modellbasis
    case ExtendedModelText = 319; // Modelltext (erweitert)
    case CombinedFuelWltp = 320; // wltp kraftstoff kombiniert
    case CurbWeight = 321; // Leergewicht
    case LicensePlate = 322; // Kennzeichen
    case TwinnerWidget = 323; // TwinnerWidget
    case LeasingResidualValueDiscount = 324; // Leasing Restwertabschlag
    case ModelKey = 325; // Modellschlüssel
    case TotalMileageFinancing = 326; // laufleistung (Finanzierung)
    case RentalVehicleActive = 327; // mietfahrzeug (aktiv)
    case DateLastMajorService = 328; // datum letzter großer kundendienst
    case MileageLastMajorService = 329; // km-Stand letzter großer kundendienst
    case DateLastMinorService = 330; // datum letzter (kleiner) kundendienst
    case MileageLastMinorService = 331; // km-Stand letzter (kleiner) kundendienst
    case DateLastOilChange = 332; // datum letzter ölwechsel
    case MileageLastOilChange = 333; // km-Stand letzter ölwechsel
    case DateLastBrakeChange = 334; // datum letzter bremsenwechsel
    case MileageLastBrakeChange = 335; // km-Stand letzter bremsenwechsel
    case FuelCityWltp = 336; // wltp kraftstoff innenstadt
    case FuelSuburbanWltp = 337; // wltp kraftstoff stadtrand
    case FuelCountryRoadWltp = 338; // wltp kraftstoff landstrasse
    case FuelHighwayWltp = 339; // wltp kraftstoff autobahn
    case CustomerOrder = 340; // kundenauftrag
    case PreSale = 341; // vorlauf
    case AdaptiveCurveLight = 342; // kurvenlicht adaptiv
    case ActionCode = 343; // aktionscode
    case SchwackeValue = 344; // schwacke wenr.
    case DatSd3FileNumber = 345; // dat sd3 aktennummer
    case LeasingSpecialConditions = 346; // leasing sonderkonditionen
    case TransportIncluded = 347; // ueberfuhrung enthalten
    case RegistrationIncluded = 348; // zulassung enthalten
    case DeliveryIncluded = 349; // kfzlieferung enthalten
    case MaintenanceIncluded = 370; // wartung enthalten
    case WearIncluded = 371; // verschleiss enthalten
    case WarrantyEndDate = 372; // garantieende datum
    case DeliveryDate = 373; // auslieferung datum
    case LeasingMatrix = 374; // leasingmatrix
    case CampaignIdentifier = 375; // kampagnenident
    case MaintenancePerMonth = 376; // Wartung pro Monat
    case WearPerMonth = 377; // Verschleiß pro Monat
    case MaintenanceAndWearCombinedPerMonth = 378; // Wartung & Verschleiß in Kombination pro Monat
    case LiabilityPerMonth = 379; // Haftplficht pro Monat
    case DeductibleLiability = 380; // Haftpflicht SB
    case ComprehensiveInsurancePerMonth = 381; // Vollkasko pro Monat
    case DeductibleComprehensive = 382; // Vollkasko SB
    case OneTimeDeliveryCost = 383; // KFZ-Lieferung einmalig
    case VehicleTaxPerMonth = 384; // KFZ-Steuer pro Monat in EUR
    case EightFoldTiresPerMonth = 385; // 8-fach Bereifung pro Monat in EUR
    case TuvPerMonth = 386; // TÜV (HU+AU) pro Monat in EUR
    case ReturnInsurancePerMonth = 387; // Rückgabeversicherung pro Monat in EUR
    case OneTimeRegistrationCost = 388; // Zulassungskosten einmalig
    case FiveZoneClimateControl = 389; // klimaautomatik 5 zonen
    case BafaBonus = 390; // Bafa Bonus (Umweltbonus Leasingmarkt)
    case Cylinders = 391; // Zylinder
    case ModelYear = 392; // Modelljahr
    case AuctionVehicle = 393; // Auktionsfahrzeug
    case PreviousOwnerName = 394; // Vorbesitzername
    case Twinner360DegreeUrl = 395; // Twinner 360Grad URL
    case VersionKey = 396; // Versionsschlüssel
    case BasicColorOutside = 397; // Grundfarbe aussen
    case RearWheelDrive = 398; // Heckantrieb
    case FrontWheelDrive = 399; // Frontantrieb
    case Gears = 400; // Anzahl Gänge
    case HeightAdjustableSteeringWheel = 401; // Lenkrad höhenverstellbar
    case MileageForExchanges = 402; // Kilometerstand für Börsen
    case MirrorsAutomaticallyDimming = 403; // Aussenspiegel autom. abblendend
    case BonusEligible = 404; // Bonus berechtigt
    case CombinedElectricConsumptionWltp = 405; // kombinierter stromverbrauch wltp
    case PreviousUse = 406; // Vornutzung
    case DatEuropeCode = 407; // DAT Europa Code
    case SupplierTradeName = 408; // Handelsname Lieferant
    case TireTypeIdentification = 409; // Reifentypkennung
    case TireDesignation = 410; // Reifenbezeichnung
    case LoadIndex = 411; // Tragfähigkeitsindex
    case SpeedIndex = 412; // Geschwindigkeitsindex
    case FuelEfficiencyClass = 413; // Kraftstoffeffklasse
    case WetGripClass = 414; // Nasshaftungsklasse
    case ExternalRollingNoiseClass = 415; // Externe Rollgeräuschklasse
    case ValueExternalRollingNoise = 416; // Wert externe Rollgeräusch
    case ExtremeSnowConditionSuitable = 418; // Extreme Schneeverhältnis geeignet
    case IceTires = 419; // Eisreifen
    case ManufacturingStartDateTireType = 420; // Herstellungsbeginn Reifentyp
    case ManufacturingEndDateTireType = 421; // Herstellungsende Reifentyp
    case TradeName = 422; // Handelsname
    case TireAdditionalInfo = 423; // Reifenzusatzinfo
    case AcChargingPower = 424; // Ladeleistung AC
    case DcChargingPower = 425; // Ladeleistung DC
    case EuTireRegisterNumberFront = 426; // EU-Reifenregisternummer (vorne)
    case IgnoreInPka = 427; // Fahrzeug im PKA ignorieren
    case KeyBoardNumber = 428; // Schlüsselbrettnummer
    case PartialLedHeadlights = 429; // Teil-Ledscheinwerfer
    case WarrantyMonths = 430; // Garantie Monate
    case ComfortSeat = 431; // Komfortsitz
    case CeramicBrake = 432; // Keramikbremse
    case SafetyIntegralSeats = 433; // Sicherheitsintegralsitze
    case AcousticGlazing = 434; // Akustikverglasung
    case ElectricParkingBrake = 435; // Elek. Parkbremse
    case HeightAdjustableDriverSeat = 436; // Fahrersitz Höhenverstellbar
    case FoldableRearSeat = 437; // Rückbank get.umleg.
    case SoftClose = 438; // Softclose
    case BrakePadIndicator = 439; // Bremsbelaganzeige
    case AudioAuxConnection = 441; // Audio-Aux-Anschluss
    case IPadIPodConnection = 442; // iPad/iPod-Anschluss
    case ClockAndTachometer = 443; // Uhr&Drehzahlm.
    case AllWheelSteering = 444; // Allradlenkung
    case MirrorsElectricallyFolding = 445; // Außenspiegel el. anklap.
    case MirrorsHeated = 446; // Außenspiegel beheizbar
    case MirrorsDimming = 447; // Außenspiegel abblend.
    case RoofRack = 448; // Dachträger
    case BlackOptic = 449; // Optikschwarz
    case PrivacyGlazing = 450; // Privacy Verglasung
    case SportsExhaustSystem = 451; // Sportabgasanlage
    case HeatProtectionGlass = 452; // Wärmeschutzglas
    case ActiveHeadRest = 453; // AktivKopfstütze
    case AutonomousDriving = 454; // autonomes Fahren
    case ModelSeries = 455; // Modellreihe
    case FinancingProduct = 456; // Finanzierungsprodukt
    case LeasingProduct = 457; // Leasingprodukt
    case MaximumPower = 458; // Motorleistung (max)
    case BatteryVoltage = 459; // Batteriespannung
    case BatteryCapacity = 460; // Batteriekapazität
    case BatteryType = 461; // Batteriebauart
    case ChargingPlug = 462; // Ladestromstecker
    case QuickdropSystem = 463; // Quickdrop-System
    case StandardCharging = 464; // Standardladung
    case StandardChargingDurationAc = 465; // Ladedauer bei Standardladung (AC) (Wallbox/Ladestation)
    case FastCharging = 466; // Schnellladung
    case FastChargingDurationDc = 467; // Ladedauer bei Schnellladung (DC) (Wallbox/Ladestation)
    case ElectricMotorRangeWltp = 468; // Elektromotor-Reichweite nach WLTP
    case ElectricMotorRangeCityWltp = 469; // Elektromotor-Reichweite innerorts nach WLTP
    case TotalRangeWltp = 470; // Gesamtreichweite nach WLTP
    case TotalRangeCityWltp = 471; // Gesamtreichweite innerorts nach WLTP
    case MobileAdId = 472; // MobileAD-ID
    case WeightedCombinedCo2EmissionsWltp = 473; // Gewichtete kombinierte CO2-Emissionen WLTP
    case WeightedCombinedElectricConsumptionWltp = 474; // Gewichteter kombinierter Stromverbrauch WLTP
    case WeightedCombinedFuelConsumptionWltp = 475; // Gewichteter kombinierter Kraftstoffverbrauch WLTP
    case DefectsInfoAccidentInfo = 476; // Mängelhinweis/Unfallinfo
    case LeasingOfferNote = 477; // Hinweis Leasingangebot
    case FinancingOfferNote = 478; // Hinweis Finanzierungsangebot
    case EuTireRegisterNumberRear = 479; // EU-Reifenregisternummer (hinten)
    case HomepageExclusive = 480; // Homepage Exklusiv
    case Youngtimer = 481; // Youngtimer
    case AboLocation = 486; // abo standort
    case AboMinimumAge = 487; // abo mindestalter in jahre
    case AboDeductible = 488; // abo selbstbeteiligung
    case AboDeposit = 489; // abo kaution
    case AboContractConditions = 490; // abo vertragsbedingungen
    case AboExtraKm = 491; // abo mehrkm
    case AboAvailability = 492; // abo verfuegbarkeit
    case AboDeliveryDate = 493; // abo lieferdatum
    case AboDeliveryPeriod = 494; // abo lieferfrist in tagen
    case AboDeliveryOptions = 495; // abo lieferoptionen
    case AboDeliveryCost = 496; // abo lieferkosten
    case AboDeliveryCostMatrix = 497; // abo lieferkosten matrix
    case AboRemark = 498; // abo bemerkung
    case AboHighlight = 499; // abo highlight
    case AboTopTags = 500; // abo toptags
    case AboPaymentConditions = 501; // abo zahlungsbedingungen
    case AboRate = 502; // abo rate
    case AboRateMatrix = 503; // abo rate matrix
    case AboStartFee = 504; // abo startgebuehr
    case AboTeaserDetailPage = 505; // abo teaser detailpage
    case InternalIdentifier = 506; // interner identifier
    case ApprovalInternetPublication = 507; // freigabe internetveröffentlichung
    case NumberOfPictures = 508; // anzahl bilder
    case HeatPump = 509; // wärmepumpe
    case Buyer = 510; // einkäufer
    case InternalInformation = 511; // interne informationen
    case OnlinePurchase = 512; // onlinekauf
    case Co2ClassWltp = 513; // co2-klasse (wltp)
    case Co2ClassWltpWeightedCombined = 514; // co2-Klasse (wltp) gewichtet, kombiniert
    case Co2ClassWltpEmptyBattery = 515; // co2-klasse (wltp) bei entladener Batterie
    case Co2EmissionsWltpEmptyBattery = 516; // co2-emissionen (wltp) bei entladener batterie
    case CombinedFuelConsumptionWltpEmptyBattery = 517; // kombinierter kraftstoffverbrauch (wltp) entl. batterie
    case BatteryStateOfHealthPercentage = 518; // state of health (soh) in % (batteriezustand)
    case VirtualVehicle = 519; // musterfahrzeug / virtuelles fahrzeug
    case ElectricityConsumptionCityWltp = 520; // stromverbrauch innenstadt wltp
    case ElectricityConsumptionSuburbanWltp = 521; // stromverbrauch stadtrand wltp
    case ElectricityConsumptionCountryRoadWltp = 522; // stromverbrauch landstrasse wltp
    case ElectricityConsumptionHighwayWltp = 523; // stromverbrauch autobahn wltp
    case FuelConsumptionEmptyBatteryCityWltp = 524; // verbrauch entladener batterie innenstadt wltp
    case FuelConsumptionEmptyBatterySuburbanWltp = 525; // verbrauch entladener batterie stadtrand wltp
    case FuelConsumptionEmptyBatteryCountryRoadWltp = 526; // verbrauch entladener batterie landstrasse wltp
    case FuelConsumptionEmptyBatteryHighwayWltp = 527; // verbrauch entladener batterie autobahn wltp
    case PkwLabelLink = 528; // link pkwlabel
    case AverageCo2PricePerTonne = 529; // Mittlerer CO2 Preis pro Tonne
    case AverageCo2PriceAccumulated = 530; // Mittlerer CO2 Preis akkumuliert
    case LowCo2PricePerTonne = 531; // Niedriger CO2 Preis pro Tonne
    case LowCo2PriceAccumulated = 532; // Niedriger CO2 Preis akkumuliert
    case HighCo2PricePerTonne = 533; // Hoher CO2 Preis pro Tonne
    case HighCo2PriceAccumulated = 534; // Hoher CO2 Preis akkumuliert
    case FuelPriceReference = 535; // Kraftstoffpreis (Referenz)
    case ElectricityPriceReference = 536; // Strompreis
    case YearOfEnergyCostsAt15000Km = 537; // Jahr der Energiekosten bei 15.000 km Jahresfahrleistung
    case EnergyCostsAt15000Km = 538; // Energiekosten bei 15.000 km Jahresfahrleistung
    case VehicleTaxPerAnnum = 539; // Kfzsteuer p.a.
    case Co2CostsStart = 540; // Zeitrahmen (von) CO2 Kosten
    case Co2CostsEnd = 541; // Zeitrahmen (bis) CO2 Kosten


    public function getName(): string {
        // TODO: replace Hardcoded values with Joomla Translations
        return match($this) {
            self::CustomerNumberOrBranch => 'Kundennummer/Filiale',
            self::InternalId => 'Interne Nummer',
            self::Category => 'Kategorie',
            self::Brand => 'Marke',
            self::Model => 'Modell',
            self::Power => 'Leistung',
            self::MainInspection => 'Hauptuntersuchung',
            self::FirstRegistration => 'Erstzulassung',
            self::Kilometers => 'Kilometer',
            self::Price => 'Preis',
            self::Vat => 'MwSt',
            self::Oldtimer => 'Oldtimer',
            self::Vin => 'Fahrzeug-Identifizierungsnummer (VIN)',
            self::DamagedVehicle => 'Beschädigtes Fahrzeug',
            self::Color => 'Farbe',
            self::ClimateControl => 'Klimaanlage/Klimaautomatik',
            self::Taxi => 'Taxi',
            self::DisabledAccessible => 'Behindertengerecht',
            self::ShowroomCar => 'Jahreswagen',
            self::NewVehicle => 'Neufahrzeug',
            self::OurRecommendation => 'Unsere Empfehlung',
            self::DealerPrice => 'Händlerpreis',
            self::Remark => 'Bemerkung',
            self::ImageId => 'Bild ID',
            self::Metallic => 'Metallic',
            self::Currency => 'Währung',
            self::VatRate => 'MwSt-Satz',
            self::Warranty => 'Garantie',
            self::AlloyWheels => 'Leichtmetallfelgen',
            self::Esp => 'ESP',
            self::Abs => 'ABS',
            self::TrailerHitch => 'Anhängerkupplung',
            self::Immobilizer => 'Wegfahrsperre',
            self::NavigationSystem => 'Navigationssystem',
            self::Sunroof => 'Schiebedach',
            self::CentralLocking => 'Zentralverriegelung',
            self::PowerWindows => 'Fensterheber',
            self::FourWheelDrive => 'Allradantrieb',
            self::Doors => 'Türen',
            self::EnvironmentalBadge => 'Umweltplakette',
            self::PowerSteering => 'Servolenkung',
            self::BiodieselSuitable => 'Biodiesel geeignet',
            self::ServiceBook => 'Scheckheftgepflegt',
            self::CatalyticConverter => 'Katalysator',
            self::Kickstarter => 'Kickstarter',
            self::EStarter => 'E-Starter',
            self::DemoVehicle => 'Vorführfahrzeug',
            self::Drive => 'Antrieb',
            self::Ccm => 'Hubraum',
            self::LoadCapacity => 'Tragkraft',
            self::Payload => 'Nutzlast',
            self::TotalWeight => 'Gesamtgewicht',
            self::LiftHeight => 'Hubhöhe',
            self::ConstructionHeight => 'Bauhöhe',
            self::YearOfConstruction => 'Baujahr',
            self::OperatingHours => 'Betriebsstunden',
            self::Seats => 'Sitzplätze',
            self::EmissionClass => 'Schadstoffklasse',
            self::CabinType => 'Kabinenart',
            self::Axles => 'Achsen',
            self::CruiseControl => 'Tempomat',
            self::AuxiliaryHeating => 'Standheizung',
            self::Cabin => 'Kabine',
            self::ProtectiveRoof => 'Schutzdach',
            self::FullFairing => 'Vollverkleidung',
            self::Municipal => 'Kommunal',
            self::Crane => 'Kran',
            self::RetarderIntarder => 'Retarder/Intarder',
            self::SleepingPlace => 'Schlafplatz',
            self::Tv => 'Fernseher',
            self::Wc => 'WC',
            self::HydraulicSystem => 'Hydraulikanlage',
            self::SlidingDoor => 'Schiebetür',
            self::AxleFormula => 'Radformel',
            self::Partition => 'Trennwand',
            self::Ebs => 'EBS',
            self::Rentable => 'Vermietbar',
            self::Compressor => 'Kompressor',
            self::AirSuspension => 'Luftfederung',
            self::DiscBrake => 'Scheibenbremse',
            self::FrontHydraulics => 'Fronthydraulik',
            self::Bss => 'BSS',
            self::QuickChange => 'Schnellwechsel',
            self::Zsa => 'ZSA',
            self::Kitchen => 'Küche',
            self::CoolingBox => 'Kühlbox',
            self::SleepingSeats => 'Schlafsitze',
            self::FrontLift => 'Frontheber',
            self::VisibleOnlyForDealers => 'Sichtbar nur für Händler',
            self::Envkv => 'ENVKV',
            self::ConsumptionCity => 'Verbrauch innerorts',
            self::ConsumptionHighway => 'Verbrauch außerorts',
            self::ConsumptionCombined => 'Verbrauch kombiniert',
            self::Emission => 'Emission',
            self::XenonHeadlights => 'Xenonscheinwerfer',
            self::SeatHeating => 'Sitzheizung',
            self::ParticulateFilter => 'Partikelfilter',
            self::ParkingAssistance => 'Einparkhilfe',
            self::SchwackeCode => 'Schwackecode',
            self::DeliveryDate => 'Lieferdatum',
            self::DeliveryPeriod => 'Lieferfrist',
            self::TransportCosts => 'Überführungskosten',
            self::HuAuNew => 'HU/AU neu',
            self::FuelType => 'Kraftstoffart',
            self::GearType => 'Getriebeart',
            self::ExportVehicle => 'Exportfahrzeug',
            self::DayRegistration => 'Tageszulassung',
            self::EyeCatcher => 'Blickfänger',
            self::Hsn => 'HSN',
            self::Tsn => 'TSN',
            self::Page1Ad => 'Seite 1 Inserat',
            self::E10Compatible => 'E10 geeignet',
            self::QualitySeal => 'Qualitätssiegel',
            self::VegetableOil => 'Pflanzenöl geeignet',
            self::Scr => 'SCR',
            self::Case => 'Koffer',
            self::CrashBar => 'Sturzbügel',
            self::Windshield => 'Scheibe',
            self::StandAirConditioning => 'Standklima',
            self::SsTire => 'S-S Bereifung',
            self::RoadRegistration => 'Straßenzulassung',
            self::BunkBed => 'Etagenbett',
            self::FixedBed => 'Festbett',
            self::RearGarage => 'Heckgarage',
            self::Awning => 'Markise',
            self::SeparateShower => 'Sep-Dusche',
            self::SolarSystem => 'Solaranlage',
            self::MiddleSeatGroup => 'Mittelsitzgruppe',
            self::RoundSeatGroup => 'Rundsitzgruppe',
            self::SideSeatGroup => 'Seitensitzgruppe',
            self::HailDamage => 'Hagelschaden',
            self::SleepingPlaces => 'Schlafplätze',
            self::VehicleLength => 'Fahrzeuglänge',
            self::VehicleWidth => 'Fahrzeugbreite',
            self::VehicleHeight => 'Fahrzeughöhe',
            self::LoadSpacePallet => 'Laderaum-Europalette',
            self::LoadVolume => 'Laderaum-Volumen',
            self::LoadLength => 'Laderaum-Länge',
            self::LoadWidth => 'Laderaum-Breite',
            self::LoadHeight => 'Laderaum-Höhe',
            self::ListingAsNew => 'Inserat als neu markieren',
            self::EffectiveAnnualRate => 'Effektiver Jahreszins',
            self::MonthlyRate => 'Monatliche Rate',
            self::Duration => 'Laufzeit',
            self::DownPayment => 'Anzahlung',
            self::FinalInstallment => 'Schlussrate',
            self::FinancingFeature => 'Finanzierungsfeature',
            self::InteriorColor => 'Interieurfarbe',
            self::InteriorType => 'Interieurtyp',
            self::Airbag => 'Airbag',
            self::PreviousOwners => 'Vorbesitzer',
            self::TopAd => 'Top Inserat',
            self::GrossCreditAmount => 'Bruttokreditbetrag',
            self::ClosingFees => 'Abschlussgebühren',
            self::RateProtection => 'Ratenabsicherung',
            self::NetCreditAmount => 'Nettokreditbetrag',
            self::ProviderBank => 'Anbieterbank',
            self::NominalInterestRate => 'Soll-zinssatz',
            self::TypeOfNominalInterestRate => 'Art des Soll-zinssatzes',
            self::CountryVersion => 'Landesversion',
            self::EnergyEfficiencyClass => 'Energieeffizienzklasse',
            self::EnvkvFuelType => 'Envkv_benzin_sorte',
            self::ElectricMirrors => 'Elektrische Seitenspiegel',
            self::SportsSuspension => 'Sportfahrwerk',
            self::SportsPackage => 'Sportpaket',
            self::Bluetooth => 'Bluetooth',
            self::OnBoardComputer => 'Bordcomputer',
            self::CdPlayer => 'CD Spieler',
            self::ElectricSeatAdjustment => 'Elektrische Sitzverstellung',
            self::TailLift => 'Ladebordwand (Heck)',
            self::SlidingDoorCommercial => 'Schiebetür (Nutzfahrzeuge)',
            self::AirSuspensionCommercial => 'Luftfederung (Nutzfahrzeuge)',
            self::SupplyDate => 'Lieferdatum',
            self::VideoUrl => 'Video-URL',
            self::HeadUpDisplay => 'Head-Up Display',
            self::HandsFreeSystem => 'Freisprechanlage',
            self::Mp3Interface => 'MP3-Schnittstelle',
            self::MultifunctionSteeringWheel => 'Multifunktionslenkrad',
            self::SkiBag => 'Skisack',
            self::TunerOrRadio => 'Tuner/Radio',
            self::SportsSeats => 'Sportsitze',
            self::PanoramicRoof => 'Panoramadach',
            self::ChildSeatAttachment => 'Kindersitzbefestigung',
            self::CorneringLight => 'Kurvenlicht',
            self::LightSensor => 'Lichtsensor',
            self::FogLights => 'Nebelscheinwerfer',
            self::DaytimeRunningLights => 'Tagfahrlicht',
            self::TractionControl => 'Traktionskontrolle',
            self::StartStopAutomatic => 'Start-Stopp-Automatik',
            self::RainSensor => 'Regensensor',
            self::NonSmokerVehicle => 'Nichtraucherfahrzeug',
            self::RoofRailing => 'Dachreling',
            self::AccidentVehicle => 'Unfallfahrzeug',
            self::Roadworthy => 'Fahrtüchtig',
            self::ProductionDate => 'Produktionsdatum',
            self::ParkingSensorsFront => 'Parkhilfe-Sensoren vorne',
            self::ParkingSensorsRear => 'Parkhilfe-Sensoren hinten',
            self::ParkingCamera => 'Rückfahrkamera',
            self::ParkingAssistanceSelfSteering => 'Parkassistenzsystem (selbstlenkend)',
            self::RotstiftPrice => 'Rotstiftpreis',
            self::EbayExport => 'eBay Export',
            self::PluginHybrid => 'Plug-in-Hybrid',
            self::CombinedElectricConsumption => 'Kombinierter Stromverbrauch',
            self::Highlight1 => 'Highlight 1',
            self::Highlight2 => 'Highlight 2',
            self::Highlight3 => 'Highlight 3',
            self::FinancingConditions => 'Finanzierungsbedingungen',
            self::HeadlightCleaningSystem => 'Scheinwerferreinigungsanlage',
            self::HighBeamAssistant => 'Fernlichtassistent',
            self::AmbientLighting => 'Ambientbeleuchtung',
            self::RearSeatHeating => 'Sitzheizung hinten',
            self::RearElectricSeatAdjustment => 'Elektrische Sitzverstellung hinten',
            self::SmokerPackage => 'Raucherpaket',
            self::FoldablePassengerSeat => 'Umklappbarer Beifahrersitz',
            self::FullDigitalCombinationInstrument => 'Volldigitales Kombiinstrument',
            self::WinterPackage => 'Winterpaket',
            self::AndroidAuto => 'Android Auto',
            self::AppleCarplay => 'Apple Carplay',
            self::InductionChargingSmartphones => 'Induktives Laden für Smartphones',
            self::IntegratedMusicStreaming => 'Musikstreaming integriert',
            self::Wifi => 'WLAN',
            self::AllSeasonTires => 'Ganzjahresreifen',
            self::SummerTires => 'Sommerreifen',
            self::WinterTires => 'Winterreifen',
            self::SpareWheel => 'Ersatzrad',
            self::PunctureKit => 'Pannenkit',
            self::FullSizeSpareWheel => 'Vollwertiges Ersatzrad',
            self::SteelWheel => 'Stahlfelgen',
            self::MissingEVBattery => 'Fehlende EV-Batterie',
            self::BoughtEVBattery => 'Gekaufte EV-Batterie',
            self::RentedEVBattery => 'Gemietete EV-Batterie',
            self::RangeExtender => 'Reichweitenverlängerer',
            self::IsofixPassenger => 'Isofix (Beifahrer)',
            self::LuggageCompartmentSeparation => 'Gepäckraumabtrennung',
            self::SwivelTowBar => 'Schwenkbare Anhängerkupplung',
            self::RemovableTowBar => 'Abnehmbare Anhängerkupplung',
            self::FixedTowBar => 'Feste Anhängerkupplung',
            self::SlidingDoorRight => 'Schiebetür rechts',
            self::SlidingDoorLeft => 'Schiebetür links',
            self::SlidingDoorsBothSides => 'Schiebetüren beidseitig',
            self::BiXenonLight => 'Bi-Xenon Licht',
            self::LaserHeadlights => 'Laser-Scheinwerfer',
            self::AlarmSystem => 'Alarmanlage',
            self::HeatedSteeringWheel => 'Beheizbares Lenkrad',
            self::AutomaticallyDimmingInteriorMirror => 'Automatisch abblendender Innenspiegel',
            self::LaneKeepingAssistant => 'Spurhalteassistent',
            self::AdaptiveCruiseControl => 'Adaptiver Tempomat',
            self::LedHeadlights => 'LED-Scheinwerfer',
            self::EmergencyBrakeAssistant => 'Notbremsassistent',
            self::ProximityWarning => 'Abstandswarner',
            self::LaneChangeAssistant => 'Spurwechselassistent',
            self::SeatVentilation => 'Sitzbelüftung',
            self::CenterArmrest => 'Mittelarmlehne',
            self::HeatedWindshield => 'Beheizbare Frontscheibe',
            self::HillStartAssist => 'Berganfahrassistent',
            self::DabRadio => 'DAB-Radio',
            self::ElectricTailgate => 'Elektrische Heckklappe',
            self::LedDaytimeRunningLight => 'LED-Tagfahrlicht',
            self::LeatherSteeringWheel => 'Lederlenkrad',
            self::LumbarSupport => 'Lendenwirbelstütze',
            self::MassageSeat => 'Massagesitz',
            self::FatigueWarningSystem => 'Müdigkeitswarnsystem',
            self::NightVisionAssistant => 'Nachtsichtassistent',
            self::EmergencyCallSystem => 'Notrufsystem',
            self::TirePressureMonitoringSystem => 'Reifendruckkontrollsystem',
            self::PaddleShifters => 'Schaltwippen',
            self::SoundSystem => 'Soundsystem',
            self::VoiceControl => 'Sprachsteuerung',
            self::Touchscreen => 'Touchscreen',
            self::Usb => 'USB',
            self::TrafficSignRecognition => 'Verkehrszeichenerkennung',
            self::KeylessCentralLocking => 'Schlüssellose Zentralverriegelung',
            self::DualZoneClimateControl => 'Klimaautomatik 2 Zonen',
            self::TripleZoneClimateControl => 'Klimaautomatik 3 Zonen',
            self::QuadrupleZoneClimateControl => 'Klimaautomatik 4 Zonen',
            self::Camera360ParkingAssistance => '360°-Kamera',
            self::NonGlaringHighBeam => 'Blendfreies Fernlicht (Matrix)',
            self::CompanyCar => 'Dienstwagen',
            self::FormerRecommendedRetailPrice => 'Ehem. UVP',
            self::EquipmentLine => 'Ausstattungslinie',
            self::ConstructionType => 'Bauart',
            self::AcceptanceDate => 'Hereinnahmedatum',
            self::MobileDeTurboAd => 'Mobile.de Turboinserat',
            self::EffectiveAnnualRateLeasing => 'Effektiver Jahreszins (Leasing)',
            self::MonthlyRateLeasing => 'Monatliche Rate (Leasing)',
            self::DurationLeasing => 'Laufzeit (Leasing)',
            self::DownPaymentLeasing => 'Anzahlung (Leasing)',
            self::FinalInstallmentLeasing => 'Schlussrate (Leasing)',
            self::TargetGroupLeasing => 'Zielgruppe (Leasing)',
            self::TotalLeasingMileage => 'Gesamtfahrleistung (Leasing)',
            self::LeasingResidualValue => 'Leasing Restwert',
            self::LeasingExtraMileageCost => 'Leasing Mehrkilometerkosten',
            self::LeasingReducedMileageCost => 'Leasing Minderkilometerkosten',
            self::ProviderBankLeasing => 'Anbieterbank (Leasing)',
            self::NominalInterestRateLeasing => 'Nominalzins (Leasing)',
            self::TypeOfNominalInterestRateLeasing => 'Art des Nominalzinses (Leasing)',
            self::ConditionsLeasing => 'Bedingungen (Leasing)',
            self::TotalLeasingAmount => 'Gesamtbetrag (Leasing)',
            self::NetCreditAmountLeasing => 'Nettokreditbetrag (Leasing)',
            self::DetailedEmissionClass => 'Detaillierte Emissionsklasse',
            self::SpeedLimitationDevice => 'Geschwindigkeitsbegrenzungsanlage',
            self::PurchasePrice => 'Einkaufspreis',
            self::RefurbishmentCosts => 'Instandsetzungskosten',
            self::MobileDeClicks => 'Mobile.de Klicks',
            self::Co2EmissionWltp => 'CO2-Emission WLTP',
            self::OpenLeads => 'Offene Leads',
            self::OpenLeadsHandler => 'Bearbeiter offener Leads',
            self::LatestOpenLeadDate => 'Letzter Lead Eingang',
            self::StorageLocation => 'Lagerplatz',
            self::ModelBase => 'Modellbasis',
            self::ExtendedModelText => 'Modelltext (erweitert)',
            self::CombinedFuelWltp => 'Kombinierter Kraftstoffverbrauch WLTP',
            self::CurbWeight => 'Leergewicht',
            self::LicensePlate => 'Kennzeichen',
            self::TwinnerWidget => 'TwinnerWidget',
            self::LeasingResidualValueDiscount => 'Leasing Restwertabschlag',
            self::ModelKey => 'Modellschlüssel',
            self::TotalMileageFinancing => 'Gesamtfahrleistung (Finanzierung)',
            self::RentalVehicleActive => 'Mietfahrzeug (aktiv)',
            self::DateLastMajorService => 'Datum letzter großer Kundendienst',
            self::MileageLastMajorService => 'Kilometerstand letzter großer Kundendienst',
            self::DateLastMinorService => 'Datum letzter kleiner Kundendienst',
            self::MileageLastMinorService => 'Kilometerstand letzter kleiner Kundendienst',
            self::DateLastOilChange => 'Datum letzter Ölwechsel',
            self::MileageLastOilChange => 'Kilometerstand letzter Ölwechsel',
            self::DateLastBrakeChange => 'Datum letzter Bremsenwechsel',
            self::MileageLastBrakeChange => 'Kilometerstand letzter Bremsenwechsel',
            self::FuelCityWltp => 'Kraftstoffverbrauch Stadt WLTP',
            self::FuelSuburbanWltp => 'Kraftstoffverbrauch Stadtrand WLTP',
            self::FuelCountryRoadWltp => 'Kraftstoffverbrauch Landstraße WLTP',
            self::FuelHighwayWltp => 'Kraftstoffverbrauch Autobahn WLTP',
            self::CustomerOrder => 'Kundenauftrag',
            self::PreSale => 'Vorverkauf',
            self::AdaptiveCurveLight => 'Adaptives Kurvenlicht',
            self::ActionCode => 'Aktionscode',
            self::SchwackeValue => 'Schwacke Wert',
            self::DatSd3FileNumber => 'Dat SD3 Aktennummer',
            self::LeasingSpecialConditions => 'Leasing Sonderkonditionen',
            self::TransportIncluded => 'Transport inklusive',
            self::RegistrationIncluded => 'Zulassung inklusive',
            self::DeliveryIncluded => 'Lieferung inklusive',
            self::MaintenanceIncluded => 'Wartung inklusive',
            self::WearIncluded => 'Verschleiß inklusive',
            self::WarrantyEndDate => 'Garantieende',
            self::LeasingMatrix => 'Leasingmatrix',
            self::CampaignIdentifier => 'Kampagnenidentifikation',
            self::MaintenancePerMonth => 'Wartung pro Monat',
            self::WearPerMonth => 'Verschleiß pro Monat',
            self::MaintenanceAndWearCombinedPerMonth => 'Wartung & Verschleiß kombiniert pro Monat',
            self::LiabilityPerMonth => 'Haftpflicht pro Monat',
            self::DeductibleLiability => 'Selbstbeteiligung Haftpflicht',
            self::ComprehensiveInsurancePerMonth => 'Vollkasko pro Monat',
            self::DeductibleComprehensive => 'Selbstbeteiligung Vollkasko',
            self::OneTimeDeliveryCost => 'Einmalige Lieferkosten',
            self::VehicleTaxPerMonth => 'Kfz-Steuer pro Monat',
            self::EightFoldTiresPerMonth => '8-fach Bereifung pro Monat',
            self::TuvPerMonth => 'TÜV pro Monat',
            self::ReturnInsurancePerMonth => 'Rückgabeversicherung pro Monat',
            self::OneTimeRegistrationCost => 'Einmalige Zulassungskosten',
            self::FiveZoneClimateControl => 'Klimaautomatik 5 Zonen',
            self::BafaBonus => 'Bafa Bonus',
            self::Cylinders => 'Zylinder',
            self::ModelYear => 'Modelljahr',
            self::AuctionVehicle => 'Auktionsfahrzeug',
            self::PreviousOwnerName => 'Name des Vorbesitzers',
            self::Twinner360DegreeUrl => 'Twinner 360 Grad URL',
            self::VersionKey => 'Versions-Schlüssel',
            self::BasicColorOutside => 'Grundfarbe außen',
            self::RearWheelDrive => 'Heckantrieb',
            self::FrontWheelDrive => 'Frontantrieb',
            self::Gears => 'Gänge',
            self::HeightAdjustableSteeringWheel => 'Lenkrad höhenverstellbar',
            self::MileageForExchanges => 'Kilometerstand für Börsen',
            self::MirrorsAutomaticallyDimming => 'Außenspiegel automatisch abblendend',
            self::BonusEligible => 'Bonus berechtigt',
            self::CombinedElectricConsumptionWltp => 'Kombinierter Stromverbrauch WLTP',
            self::PreviousUse => 'Vornutzung',
            self::DatEuropeCode => 'DAT Europa Code',
            self::SupplierTradeName => 'Handelsname Lieferant',
            self::TireTypeIdentification => 'Reifentypkennung',
            self::TireDesignation => 'Reifenbezeichnung',
            self::LoadIndex => 'Tragfähigkeitsindex',
            self::SpeedIndex => 'Geschwindigkeitsindex',
            self::FuelEfficiencyClass => 'Kraftstoffeffizienzklasse',
            self::WetGripClass => 'Nasshaftungsklasse',
            self::ExternalRollingNoiseClass => 'Externe Rollgeräuschklasse',
            self::ValueExternalRollingNoise => 'Wert externes Rollgeräusch',
            self::ExtremeSnowConditionSuitable => 'Extreme Schneeverhältnisse geeignet',
            self::IceTires => 'Eisreifen',
            self::ManufacturingStartDateTireType => 'Herstellungsbeginn Reifentyp',
            self::ManufacturingEndDateTireType => 'Herstellungsende Reifentyp',
            self::TradeName => 'Handelsname',
            self::TireAdditionalInfo => 'Reifenzusatzinfo',
            self::AcChargingPower => 'Ladeleistung AC',
            self::DcChargingPower => 'Ladeleistung DC',
            self::EuTireRegisterNumberFront => 'EU-Reifenregisternummer (vorne)',
            self::IgnoreInPka => 'Fahrzeug im PKA ignorieren',
            self::KeyBoardNumber => 'Schlüsselbrettnummer',
            self::PartialLedHeadlights => 'Teil-LED-Scheinwerfer',
            self::WarrantyMonths => 'Garantie Monate',
            self::ComfortSeat => 'Komfortsitz',
            self::CeramicBrake => 'Keramikbremse',
            self::SafetyIntegralSeats => 'Sicherheitsintegralsitze',
            self::AcousticGlazing => 'Akustikverglasung',
            self::ElectricParkingBrake => 'Elektronische Parkbremse',
            self::HeightAdjustableDriverSeat => 'Fahrersitz höhenverstellbar',
            self::FoldableRearSeat => 'Rückbank umlegbar',
            self::SoftClose => 'Softclose',
            self::BrakePadIndicator => 'Bremsbelaganzeige',
            self::AudioAuxConnection => 'Audio-Aux-Anschluss',
            self::IPadIPodConnection => 'iPad/iPod-Anschluss',
            self::ClockAndTachometer => 'Uhr & Drehzahlmesser',
            self::AllWheelSteering => 'Allradlenkung',
            self::MirrorsElectricallyFolding => 'Außenspiegel el. anklappbar',
            self::MirrorsHeated => 'Außenspiegel beheizbar',
            self::MirrorsDimming => 'Außenspiegel abblendend',
            self::RoofRack => 'Dachträger',
            self::BlackOptic => 'Optikschwarz',
            self::PrivacyGlazing => 'Privacy Verglasung',
            self::SportsExhaustSystem => 'Sportabgasanlage',
            self::HeatProtectionGlass => 'Wärmeschutzglas',
            self::ActiveHeadRest => 'Aktive Kopfstützen',
            self::AutonomousDriving => 'Autonomes Fahren',
            self::ModelSeries => 'Modellreihe',
            self::FinancingProduct => 'Finanzierungsprodukt',
            self::LeasingProduct => 'Leasingprodukt',
            self::MaximumPower => 'Maximale Leistung',
            self::BatteryVoltage => 'Batteriespannung',
            self::BatteryCapacity => 'Batteriekapazität',
            self::BatteryType => 'Batterietyp',
            self::ChargingPlug => 'Ladestecker',
            self::QuickdropSystem => 'Quickdrop-System',
            self::StandardCharging => 'Standardladung',
            self::StandardChargingDurationAc => 'Standardladedauer AC',
            self::FastCharging => 'Schnellladung',
            self::FastChargingDurationDc => 'Schnellladedauer DC',
            self::ElectricMotorRangeWltp => 'Elektromotor-Reichweite WLTP',
            self::ElectricMotorRangeCityWltp => 'Elektromotor-Reichweite Stadt WLTP',
            self::TotalRangeWltp => 'Gesamtreichweite WLTP',
            self::TotalRangeCityWltp => 'Gesamtreichweite Stadt WLTP',
            self::MobileAdId => 'MobileAD-ID',
            self::WeightedCombinedCo2EmissionsWltp => 'Gewichtete kombinierte CO2-Emissionen WLTP',
            self::WeightedCombinedElectricConsumptionWltp => 'Gewichteter kombinierter Stromverbrauch WLTP',
            self::WeightedCombinedFuelConsumptionWltp => 'Gewichteter kombinierter Kraftstoffverbrauch WLTP',
            self::DefectsInfoAccidentInfo => 'Mängelhinweis/Unfallinfo',
            self::LeasingOfferNote => 'Hinweis Leasingangebot',
            self::FinancingOfferNote => 'Hinweis Finanzierungsangebot',
            self::EuTireRegisterNumberRear => 'EU-Reifenregisternummer (hinten)',
            self::HomepageExclusive => 'Homepage Exklusiv',
            self::Youngtimer => 'Youngtimer',
            self::AboLocation => 'Abo Standort',
            self::AboMinimumAge => 'Abo Mindestalter',
            self::AboDeductible => 'Abo Selbstbeteiligung',
            self::AboDeposit => 'Abo Kaution',
            self::AboContractConditions => 'Abo Vertragsbedingungen',
            self::AboExtraKm => 'Abo Mehrkilometer',
            self::AboAvailability => 'Abo Verfügbarkeit',
            self::AboDeliveryDate => 'Abo Lieferdatum',
            self::AboDeliveryPeriod => 'Abo Lieferfrist',
            self::AboDeliveryOptions => 'Abo Lieferoptionen',
            self::AboDeliveryCost => 'Abo Lieferkosten',
            self::AboDeliveryCostMatrix => 'Abo Lieferkostenmatrix',
            self::AboRemark => 'Abo Bemerkung',
            self::AboHighlight => 'Abo Highlight',
            self::AboTopTags => 'Abo Top-Tags',
            self::AboPaymentConditions => 'Abo Zahlungsbedingungen',
            self::AboRate => 'Abo Rate',
            self::AboRateMatrix => 'Abo Ratenmatrix',
            self::AboStartFee => 'Abo Startgebühr',
            self::AboTeaserDetailPage => 'Abo Teaser Detailseite',
            self::InternalIdentifier => 'Interner Identifier',
            self::ApprovalInternetPublication => 'Freigabe Internetveröffentlichung',
            self::NumberOfPictures => 'Anzahl der Bilder',
            self::HeatPump => 'Wärmepumpe',
            self::Buyer => 'Einkäufer',
            self::InternalInformation => 'Interne Informationen',
            self::OnlinePurchase => 'Onlinekauf',
            self::Co2ClassWltp => 'CO2-Klasse (WLTP)',
            self::Co2ClassWltpWeightedCombined => 'CO2-Klasse (WLTP) gewichtet, kombiniert',
            self::Co2ClassWltpEmptyBattery => 'CO2-Klasse (WLTP) bei entladener Batterie',
            self::Co2EmissionsWltpEmptyBattery => 'CO2-Emissionen (WLTP) bei entladener Batterie',
            self::CombinedFuelConsumptionWltpEmptyBattery => 'Kombinierter Kraftstoffverbrauch (WLTP) entladene Batterie',
            self::BatteryStateOfHealthPercentage => 'State of Health (SOH) in % (Batteriezustand)',
            self::VirtualVehicle => 'Musterfahrzeug / Virtuelles Fahrzeug',
            self::ElectricityConsumptionCityWltp => 'Stromverbrauch Innenstadt WLTP',
            self::ElectricityConsumptionSuburbanWltp => 'Stromverbrauch Stadtrand WLTP',
            self::ElectricityConsumptionCountryRoadWltp => 'Stromverbrauch Landstraße WLTP',
            self::ElectricityConsumptionHighwayWltp => 'Stromverbrauch Autobahn WLTP',
            self::FuelConsumptionEmptyBatteryCityWltp => 'Verbrauch entladener Batterie Innenstadt WLTP',
            self::FuelConsumptionEmptyBatterySuburbanWltp => 'Verbrauch entladener Batterie Stadtrand WLTP',
            self::FuelConsumptionEmptyBatteryCountryRoadWltp => 'Verbrauch entladener Batterie Landstraße WLTP',
            self::FuelConsumptionEmptyBatteryHighwayWltp => 'Verbrauch entladener Batterie Autobahn WLTP',
            self::PkwLabelLink => 'Link PKW-Label',
        };
    }

    public function getTranslationKey()
    {
        $input = $this->name;
        $result = '';

        for ($i = 0; $i < strlen($input); $i++) {
            $char = $input[$i];

            if (ctype_upper($char)) {
                $result .= '_' . strtoupper($char);
            } else {
                $result .= strtoupper($char);
            }
        }

        return ltrim($result, '_');
    }

    public function convertValue($value): mixed
    {
        if (!$value) return $value;

        return match ($this) {
            self::EmissionClass => (function(int $emission_class) {
                return "Euro {$emission_class}";
            })(intval($value)),
            self::DetailedEmissionClass => (function(string $emission_class) {
                $parts = explode('_', $emission_class);
                $without_prefix = strtolower(str_ireplace('euro', '', (array_shift($parts))));

                return implode('-', array_merge(["Euro $without_prefix"], $parts));
            })($value),
            self::EnvironmentalBadge => (function(EnvironmentalBadge $badge) {
                return $badge->getTranslationKey();
            })(EnvironmentalBadge::tryFrom(intval($value))),
            self::FuelType => (function(FuelType $fuelType) {
                return $fuelType->getTranslationKey();
            })(FuelType::tryFrom(intval($value))),
            self::GearType => (function(GearType $gearType) {
                return $gearType->getTranslationKey();
            })(GearType::tryFrom(intval($value))),
            self::InteriorColor => (function(InteriorColor $interiorColor) {
                return $interiorColor->getTranslationKey();
            })(InteriorColor::tryFrom(intval($value))),
            self::BasicColorOutside => BaseColor::getTranslationKey(strtolower($value)),
            self::InteriorType => (function(InteriorType $interiorType) {
                return $interiorType->getTranslationKey();
            })(InteriorType::tryFrom(intval($value))),
            self::Airbag => (function(AirbagOption $airbag) {
                return $airbag->getTranslationKey();
            })(AirbagOption::tryFrom(intval($value))),
            self::VatRate => (function(float | null $val): float | null {
                return $val / 100;
            })(floatval($value)),
            default => $value,
        };
    }

    public function getFieldInstance($dataset)
    {
        return new Field($this, $dataset);
    }
}

class VehicleUtility {
    // General Information related to description and images
    const technicalInformation = [
        VehicleData::Ccm,
        VehicleData::Doors,
        VehicleData::Seats,
        VehicleData::PreviousOwners,
        VehicleData::Color,
        VehicleData::InteriorType,
        VehicleData::InteriorColor,
    ];

    // Interior & Comfort features
    const interiorComfort = [
        VehicleData::SeatHeating,
        VehicleData::RearSeatHeating,
        VehicleData::ElectricSeatAdjustment,
        VehicleData::RearElectricSeatAdjustment,
        VehicleData::HeatedSteeringWheel,
        VehicleData::AirSuspension,
        VehicleData::CenterArmrest,
        VehicleData::MultifunctionSteeringWheel,
        VehicleData::ComfortSeat,
        VehicleData::MassageSeat,
        VehicleData::FoldablePassengerSeat,
        VehicleData::FoldableRearSeat,
        VehicleData::ClimateControl,
        VehicleData::SeatVentilation,
        VehicleData::DualZoneClimateControl,
        VehicleData::TripleZoneClimateControl,
        VehicleData::QuadrupleZoneClimateControl,
        VehicleData::FiveZoneClimateControl,
        VehicleData::NonSmokerVehicle,
        VehicleData::FullDigitalCombinationInstrument,
        VehicleData::LeatherSteeringWheel,
        VehicleData::AmbientLighting,
        VehicleData::IsofixPassenger,
        VehicleData::LuggageCompartmentSeparation,
        VehicleData::StandAirConditioning,
        VehicleData::InductionChargingSmartphones,
    ];

    const audioCommunication = [
        VehicleData::AndroidAuto,
        VehicleData::Bluetooth,
        VehicleData::DabRadio,
        VehicleData::IntegratedMusicStreaming,
        VehicleData::Usb,
        VehicleData::AppleCarplay,
        VehicleData::OnBoardComputer,
        VehicleData::Mp3Interface,
        VehicleData::VoiceControl,
        VehicleData::TunerOrRadio,
        VehicleData::Touchscreen,
        VehicleData::SoundSystem,
        VehicleData::CdPlayer,
        VehicleData::Wifi,
        VehicleData::Tv,
    ];

    // Safety & Assistance features
    const safetyAssistanceExterior = [
        VehicleData::Abs,
        VehicleData::AdaptiveCruiseControl,
        VehicleData::Airbag,
        VehicleData::EmergencyBrakeAssistant,
        VehicleData::Esp,
        VehicleData::LaneChangeAssistant,
        VehicleData::LaneKeepingAssistant,
        VehicleData::NightVisionAssistant,
        VehicleData::ParkingAssistance,
        VehicleData::ParkingAssistanceSelfSteering,
        VehicleData::ParkingCamera,
        VehicleData::ParkingSensorsFront,
        VehicleData::ParkingSensorsRear,
        VehicleData::Camera360ParkingAssistance,
        VehicleData::ProximityWarning,
        VehicleData::TirePressureMonitoringSystem,
        VehicleData::TrafficSignRecognition,
        VehicleData::LedHeadlights,
        VehicleData::FogLights,
        VehicleData::PanoramicRoof,
        VehicleData::Sunroof,
        VehicleData::PrivacyGlazing,
        VehicleData::RoofRack,
        VehicleData::BiXenonLight,
        VehicleData::LaserHeadlights,
        VehicleData::DaytimeRunningLights,
        VehicleData::Color,
        VehicleData::PowerWindows,
        VehicleData::EmergencyCallSystem,
        VehicleData::CentralLocking,
        VehicleData::SpeedLimitationDevice,
        VehicleData::RainSensor,
        VehicleData::TractionControl,
        VehicleData::AlloyWheels,
        VehicleData::NonGlaringHighBeam,
        VehicleData::ChildSeatAttachment,
        VehicleData::HillStartAssist,
        VehicleData::ElectricMirrors,
        VehicleData::MirrorsElectricallyFolding,
        VehicleData::MirrorsHeated,
        VehicleData::MirrorsDimming,
        VehicleData::AllWheelSteering,
        VehicleData::RoofRailing,
        VehicleData::ElectricTailgate,
        VehicleData::TrailerHitch,
        VehicleData::FourWheelDrive,
        VehicleData::Immobilizer,
        VehicleData::LedDaytimeRunningLight,
        VehicleData::WinterPackage,
        VehicleData::PunctureKit,
        VehicleData::SpareWheel,
        VehicleData::WinterTires,
        VehicleData::SummerTires,
        VehicleData::AllSeasonTires,
        VehicleData::SwivelTowBar,
        VehicleData::RemovableTowBar,
        VehicleData::FixedTowBar,
        VehicleData::ServiceBook,
        VehicleData::CeramicBrake,
        VehicleData::HeatedWindshield,
        VehicleData::HeatProtectionGlass,
        VehicleData::SportsExhaustSystem,
    ];

    // Equipment features that have binary options
    const equipment = [
        VehicleData::MassageSeat,
        VehicleData::NavigationSystem,
        VehicleData::NavigationSystem,
        VehicleData::PanoramicRoof,
        VehicleData::PowerWindows,
        VehicleData::PrivacyGlazing,
        VehicleData::QuadrupleZoneClimateControl,
        VehicleData::RearElectricSeatAdjustment,
        VehicleData::RoofRack,
        VehicleData::Scr,
        VehicleData::ShowroomCar,
        VehicleData::Sunroof,
        VehicleData::TirePressureMonitoringSystem,
        VehicleData::TrailerHitch,
        VehicleData::TripleZoneClimateControl,
        VehicleData::Usb,
        VehicleData::Wifi,
        VehicleData::SportsSeats,
        VehicleData::LaneKeepingAssistant,
        VehicleData::LaneChangeAssistant,
        VehicleData::FullDigitalCombinationInstrument,
        VehicleData::SeatHeating,
        VehicleData::SeatVentilation,
        VehicleData::VoiceControl,
        VehicleData::RainSensor,
        VehicleData::CenterArmrest,
        VehicleData::HillStartAssist,
        VehicleData::ParkingAssistanceSelfSteering,
    ];
}


class Field {
    public mixed $value;
    public string $name;
    public string $_tKey;

    const non_nummeric = [
        VehicleData::FirstRegistration,
        VehicleData::Model,
        VehicleData::ModelBase,
        VehicleData::Brand,
        VehicleData::InternalId,
        VehicleData::ImageId,
        VehicleData::Vin,
        VehicleData::YearOfEnergyCostsAt15000Km,
        VehicleData::Co2CostsStart,
        VehicleData::Co2CostsEnd,
    ];

    public function __construct(VehicleData $field, $dataset) {
        $this->name = $field->getName();
        $this->value = self::getValue($field, $dataset);

        $this->_tKey = $field->getTranslationKey();
    }

    public static function getValue(VehicleData $field, $dataset)
    {
        $value = $field->convertValue($dataset[$field->value]);

        if (empty(trim($value))) {
            return null;
        }

        if (is_numeric($value) && !in_array($field, self::non_nummeric)) {
            $value = floatval($value);
        }

        return $value;
    }
}

class Eco {
    public string $fuel_type;
    public string $fuel_unit = 'l/100km';

    public ?float $fuel_consumption_combined = null;
    public ?float $fuel_consumption_combined_weighted = null;
    public ?float $fuel_consumption_city = null;
    public ?float $fuel_consumption_suburban = null;
    public ?float $fuel_consumption_countryside = null;
    public ?float $fuel_consumption_highway = null;

    // Only used for Hybrids
    public ?float $fuel_consumption_empty_battery = null;
    public ?float $fuel_consumption_empty_battery_city = null;
    public ?float $fuel_consumption_empty_battery_suburban = null;
    public ?float $fuel_consumption_empty_battery_countryside = null;
    public ?float $fuel_consumption_empty_battery_highway = null;

    public ?float $energy_consumption_combined = null;
    public ?float $energy_consumption_combined_weighted = null;
    public ?float $energy_consumption_city = null;
    public ?float $energy_consumption_suburban = null;
    public ?float $energy_consumption_countryside = null;
    public ?float $energy_consumption_highway = null;

    public string $co2_label;
    public ?string $co2_label_empty_battery = null;

    public float $co2_emission;
    public ?float $co2_emission_empty_battery = null;

    public ?string $badge = null;

    public ?string $emission_class;


    public ?float $range = null;

    public ?int $co2_cost_start = null;
    public ?int $co2_cost_end = null;
    public ?float $fuel_price = null;
    public ?float $energy_price = null;
    public ?float $co2_cost_low_per_t = null;
    public ?float $co2_cost_low_accumulated = null;
    public ?float $co2_cost_high_per_t = null;
    public ?float $co2_cost_high_accumulated = null;
    public ?float $co2_cost_average_per_t = null;
    public ?float $co2_cost_average_accumulated = null;

    public ?float $energy_cost_at_15000km = null;
    public ?int $energy_cost_at_15000km_year = null;

    public function __construct(array $data)
    {
        $fuelType = FuelType::tryFrom($data[VehicleData::FuelType->value]);
        $this->fuel_type = $fuelType->getTranslationKey();

        $this->badge = Field::getValue(VehicleData::EnvironmentalBadge, $data);

        $this->energy_cost_at_15000km = parse_float($data[VehicleData::EnergyCostsAt15000Km->value]);
        $this->energy_cost_at_15000km_year = Field::getValue(VehicleData::YearOfEnergyCostsAt15000Km, $data);

        $this->fuel_price = parse_float($data[VehicleData::FuelPriceReference->value]);
        $this->energy_price = parse_float($data[VehicleData::ElectricityPriceReference->value]);
        $this->co2_cost_start = parse_float($data[VehicleData::Co2CostsStart->value]);
        $this->co2_cost_end = parse_float($data[VehicleData::Co2CostsEnd->value]);

        $this->co2_cost_low_per_t = parse_float($data[VehicleData::LowCo2PricePerTonne->value]);
        $this->co2_cost_low_accumulated = parse_float($data[VehicleData::LowCo2PriceAccumulated->value]);
        $this->co2_cost_average_per_t = parse_float($data[VehicleData::AverageCo2PricePerTonne->value]);
        $this->co2_cost_average_accumulated = parse_float($data[VehicleData::AverageCo2PriceAccumulated->value]);
        $this->co2_cost_high_per_t = parse_float($data[VehicleData::HighCo2PricePerTonne->value]);
        $this->co2_cost_high_accumulated = parse_float($data[VehicleData::HighCo2PriceAccumulated->value]);

        $this->co2_label = $data[VehicleData::Co2ClassWltp->value];
        if (empty($this->co2_label)) {
            $this->co2_label = $data[VehicleData::Co2ClassWltpWeightedCombined->value];
        }

        $this->emission_class = Field::getValue(VehicleData::DetailedEmissionClass, $data);
        if (empty($this->emission_class)) {
            $this->emission_class = Field::getValue(VehicleData::EmissionClass, $data);
        }

        $this->co2_emission = (float) $data[VehicleData::Co2EmissionWltp->value];

        if ($fuelType === FuelType::Electric || $fuelType === FuelType::Hybrid) {
            $this->energy_consumption_combined = parse_float($data[VehicleData::CombinedElectricConsumptionWltp->value]);
            $this->energy_consumption_city = parse_float($data[VehicleData::ElectricityConsumptionCityWltp->value]);
            $this->energy_consumption_suburban = parse_float($data[VehicleData::ElectricityConsumptionSuburbanWltp->value]);
            $this->energy_consumption_countryside = parse_float($data[VehicleData::ElectricityConsumptionCountryRoadWltp->value]);
            $this->energy_consumption_highway = parse_float($data[VehicleData::ElectricityConsumptionHighwayWltp->value]);

            $this->range = parse_float($data[VehicleData::ElectricMotorRangeWltp->value]);

            if ($fuelType === FuelType::Hybrid) {
                $this->energy_consumption_combined_weighted = parse_float($data[VehicleData::WeightedCombinedElectricConsumptionWltp->value]);
                $this->fuel_consumption_combined_weighted = parse_float($data[VehicleData::WeightedCombinedFuelConsumptionWltp->value]);

                $this->fuel_consumption_empty_battery = parse_float($data[VehicleData::CombinedFuelConsumptionWltpEmptyBattery->value]);
                $this->fuel_consumption_empty_battery_city = parse_float($data[VehicleData::FuelConsumptionEmptyBatteryCityWltp->value]);
                $this->fuel_consumption_empty_battery_suburban = parse_float($data[VehicleData::FuelConsumptionEmptyBatterySuburbanWltp->value]);
                $this->fuel_consumption_empty_battery_countryside = parse_float($data[VehicleData::FuelConsumptionEmptyBatteryCountryRoadWltp->value]);
                $this->fuel_consumption_empty_battery_highway = parse_float($data[VehicleData::FuelConsumptionEmptyBatteryHighwayWltp->value]);

                $this->co2_emission_empty_battery = parse_float($data[VehicleData::Co2EmissionsWltpEmptyBattery->value]);
                $this->co2_label_empty_battery = $data[VehicleData::Co2ClassWltpEmptyBattery->value];
            }
        } else {
            $this->fuel_consumption_combined = parse_float($data[VehicleData::CombinedFuelWltp->value]);
            $this->fuel_consumption_city = parse_float($data[VehicleData::FuelCityWltp->value]);
            $this->fuel_consumption_suburban = parse_float($data[VehicleData::FuelSuburbanWltp->value]);
            $this->fuel_consumption_countryside = parse_float($data[VehicleData::FuelCountryRoadWltp->value]);
            $this->fuel_consumption_highway = parse_float($data[VehicleData::FuelHighwayWltp->value]);
        }

        if ($fuelType === FuelType::CNG || $fuelType === FuelType::LPG) {
            $this->fuel_unit = 'kg/100km';
        }
    }
}


class Product {
    public $id;
    public $dealer_internal_id;
    public $status = 1;
    public $condition;
    public $manufacturer;
    public $model;
    public $price;
    public $vat;
    public $power;
    public $ccm;
    public $free_text;
    public $category;
    public $recommended;

    public $equipment = [];
    public $additional = [];
    public $dealer;
    public $images = [];

    public $title;
    public $mileage;
    public $fuel_type;
    public $firstRegistration;
    public $preowners;
    public $gearType;
    public $eco;
    public $climate;
    public $doors;
    public $seats;
    public $color;
    public $interior_color;
    public $interior_type;
    public $base_color;
    public $safetyAssistanceExterior = [];
    public $comfortInterior = [];
    public $audioCommunication = [];

    private $dataset = [];

    public function __construct(array $data, array $images)
    {
        $this->dataset = $data;

        $this->dealer_internal_id = $this->getFieldValue(VehicleData::InternalId);
        // We need to hash the ID, because it's not uniform
        $this->id = hexdec(substr(sha1($this->dealer_internal_id), 0, 10));

        $this->model = $this->getFieldValue(VehicleData::ModelBase);

        $this->manufacturer = new Manufacturer($data);
        $this->price = new Price($data);
        $this->vat = new Vat($data);
        $this->category = new Category($data);
        $this->recommended = $this->getFieldValue(VehicleData::OurRecommendation);

        // TODO: Hanlde Deadlers better
        $this->dealer = [
            'city' => 'Undefined',
            'zip' => '00000',
        ];

        $this->mileage = new Mileage($data);
        $this->fuel_type = $this->getFieldValue(VehicleData::FuelType);
        $this->firstRegistration = $this->getFieldValue(VehicleData::FirstRegistration);
        $this->preowners = $this->getFieldValue(VehicleData::PreviousOwners);
        $this->gearType = $this->getFieldValue(VehicleData::GearType);
        $this->power = new Power($data);
        $this->ccm = $this->getFieldValue(VehicleData::Ccm);
        $this->climate = Climate::getValue($data);

        $this->seats = $this->getFieldValue(VehicleData::Seats);
        $this->doors = $this->getFieldValue(VehicleData::Doors);

        $this->color = $this->getFieldValue(VehicleData::Color);
        $this->base_color = $this->getFieldValue(VehicleData::BasicColorOutside);
        $this->interior_color = $this->getFieldValue(VehicleData::InteriorColor);
        $this->interior_type = $this->getFieldValue(VehicleData::InteriorType);

        $this->free_text = $this->getFieldValue(VehicleData::Remark);

        $this->condition = new Condition($data);

        foreach (VehicleUtility::equipment as $equipment) {
            $field = $this->getField($equipment);

            if ($field->value == 1) {
                $this->equipment[] = $field->name;
            }
        }

        $imageId = $this->dealer_internal_id;

        $this->images = array_map(function($image) {
            return (object) [
                'href' => "/demouploads/$image",
            ];
        }, array_values(array_filter($images, function ($image) use($imageId) {
            return substr($image, 0, strrpos($image, '_')) === $imageId;
        })));

        // Mobile.de extras
        $this->title = implode(' ', [$this->manufacturer->value, $this->getFieldValue(VehicleData::Model)]);
        $this->eco = new Eco($data);

        foreach (array_unique(VehicleUtility::safetyAssistanceExterior) as $item) {
            $field = $this->getField($item);

            if ($field->value == '' || $field->value == 0) continue;

            $this->safetyAssistanceExterior[] = strtolower($field->_tKey);
        }

        foreach (array_unique(VehicleUtility::audioCommunication) as $item) {
            $field = $this->getField($item);

            if ($field->value == '' || $field->value == 0) continue;

            $this->audioCommunication[] = strtolower($field->_tKey);
        }

        foreach (array_unique(VehicleUtility::interiorComfort) as $item) {
            $field = $this->getField($item);

            if ($field->value == '' || $field->value == 0) continue;

            $this->comfortInterior[] = strtolower($field->_tKey);
        }
    }

    private function getField(VehicleData $field): Field | null
    {
        return new Field($field, $this->dataset);
    }

    private function getFieldValue(VehicleData $field): mixed
    {
        return Field::getValue($field, $this->dataset);
    }

    public function getData()
    {
       return $this->dataset;
    }

    public function getSortValue()
    {
        return 0;
    }

    public function getVin()
    {
        return $this->getFieldValue(VehicleData::Vin);
    }
}



class Pixelcms_articlesModelMobile_csv_products  extends BaseModel
{
    protected $module = 'mobile_csv';
    protected $api_key;
    protected $articleId;
    protected $customer_id;
    protected $contact_persons;
    protected $password;
    protected $username;
    protected $language;
    protected $url;
    protected $ads;
    protected $allParams;
    protected $menuId;
    protected $filters = [];

    protected $defaults = [
        'print_btn' => 1,
        'price_style' => 0,
        'speed', 750,
        'disable_title_image' => 0,
    ];

    protected $hidden = [
        'tracktorpool_apikey',
        'tracktorpool_customer_id',
        'tracktorpool_username',
        'tracktorpool_password',
        'traktorpool_condition',
        'traktorpool_status',
    ];

    public function __construct(array $config = array())
    {
        $lang = JFactory::getLanguage();
        $langCode = $lang->getTag();
        $langCodeShort = substr($langCode, 0, 2);

        parent::__construct($config);

        /** @var ExtendedRegistry $params - Settings for current Menu Item */
        $params = $this->getState('params');

        $this->menuId = $params->getInt('Itemid');

        $this->api_key = $params->get('tracktorpool_apikey');
        $this->customer_id = $params->get('tracktorpool_customer_id');

        //$this->print_btn = $jinput->getInt('print_btn');
        $this->articleId = $params->get('articleId', null);

        $this->language = $langCode;
    }

    public function getRelatedAds()
    {
        $ads = $this->getAds();
        $ad = $this->getData();
        $relatedAds =  array_values(array_filter($ads, function ($relatedAd) use ($ad) {
            return !!($ad->category->id == $relatedAd->category->id && $ad->id != $relatedAd->id);
        }));

        self::fisherYatesShuffle($relatedAds, $ad->id);

        return $relatedAds;
    }

    private function getAds()
    {
        $ads = self::getCachedAds();

        foreach ($ads as $ad) {
            $query = http_build_query(array_merge($this->params->toArray(), [
                'articleId' => $ad->id,
                'Itemid' => $this->menuId,
                'alias' => $ad->title,
            ]));
            $ad->link = JRoute::_("index.php?$query");
        }

        return $ads;
    }

    private function getCachedAds($force = false)
    {
        $cacheKey = "csv_products.$this->customer_id";
        $ads = $this->cache->get($cacheKey, $this->module);

        if (empty($ads) || $force) {
            $ads = $this->fetchAds();
            $this->cache->store( serialize($ads), $cacheKey, $this->module);
        } else {
            $ads = unserialize($ads);
        }

        return $ads ?? [];
    }

    /**
     *
     * @return mixed
     *
     * @since version
     * @throws Exception
     */
    private function fetchAds()
    {
        $file = file(JPATH_BASE . '/demouploads/export.csv');

        $images = array_filter(scandir(JPATH_BASE . '/demouploads'), function ($file) {
            return str_ends_with($file, 'jpg');
        });

        return array_map(function($line) use($images) {
            $data = str_getcsv(mb_convert_encoding($line, "UTF-8", "Windows-1252"), ';');
            return new Product($data, $images);
        }, array_splice($file, 1));
    }

    public function getData()
    {
        $data = $this->getAds();

        if ($this->articleId) {
            $key = array_search($this->articleId, array_column($data, 'id'));

            if ($key === false) {
                throw new \Exception("Fahrzeug nicht gefunden", 404);
            }

            $data = $data[$key];
        }

        return $data;
    }

    private static function fisherYatesShuffle(&$items, $seed)
    {
        if (is_string($seed)) {
            mt_srand(crc32($seed));
        } else {
            mt_srand($seed);
        }

        $items = array_values($items);
        for ($i = count($items) - 1; $i > 0; $i--)
        {
            $j = mt_rand(0, $i);
            $tmp = $items[$i];
            $items[$i] = $items[$j];
            $items[$j] = $tmp;
        }
    }

    public function getParams()
    {
        return $this->params;
    }
}
